# Copyright 2004-2013 Tom Rothamel <pytom@bishoujo.us>
#
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge,
# publish, distribute, sublicense, and/or sell copies of the Software,
# and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

# This file contains the AST for the Ren'Py script language. Each class
# here corresponds to a statement in the script language.

# NOTE:
# When updating this file, consider if lint.py or warp.py also need
# updating.

import renpy.display

import re
import time
import md5

def next_node(n):
    """
    Indicates the next node that should be executed. When a statement 
    can crash, this should be set as early as possible, so that ignore
    can bring us there.
    """

    renpy.game.context().next_node = n

class ParameterInfo(object):
    """
    This class is used to store information about parameters to a
    label.
    """
    def __init__(self, parameters, positional, extrapos, extrakw):

        # A list of parameter name, default value pairs.
        self.parameters = parameters

        # A list, giving the positional parameters to this function,
        # in order.
        self.positional = positional

        # A variable that takes the extra positional arguments, if
        # any. None if no such variable exists.
        self.extrapos = extrapos

        # A variable that takes the extra keyword arguments, if
        # any. None if no such variable exists.
        self.extrakw = extrakw

class ArgumentInfo(object):

    def __init__(self, arguments, extrapos, extrakw):

        # A list of (keyword, expression) pairs. If an argument doesn't
        # have a keyword, it's thought of as positional.
        self.arguments = arguments

        # An expression giving extra positional arguments being
        # supplied to this function.
        self.extrapos = extrapos

        # An expression giving extra keyword arguments that need
        # to be supplied to this function.
        self.extrakw = extrakw


def __newobj__(cls, *args):
    return cls.__new__(cls, *args)  

# This represents a string containing python code.
class PyExpr(unicode):

    __slots__ = [ 
        'filename',
        'linenumber',
        ]

    def __new__(cls, s, filename, linenumber):
        self = unicode.__new__(cls, s)
        self.filename = filename
        self.linenumber = linenumber
        return self

    def __getnewargs__(self):
        return (unicode(self), self.filename, self.linenumber) # E1101
    
class PyCode(object):

    __slots__ = [
        'source',
        'location',
        'mode',
        'bytecode',
        'hash',
        ]

    def __getstate__(self):
        return (1, self.source, self.location, self.mode)

    def __setstate__(self, state):
        (_, self.source, self.location, self.mode) = state
        self.bytecode = None

        if renpy.game.script.record_pycode:
            renpy.game.script.all_pycode.append(self)

    def __init__(self, source, loc=('<none>', 1), mode='exec'):

        if isinstance(source, PyExpr):
            loc = (source.filename, source.linenumber, source)

        # The source code.
        self.source = source

        # The time is necessary so we can disambiguate between Python
        # blocks on the same line in different script versions.
        self.location = loc + ( int(time.time()), )
        self.mode = mode

        # This will be initialized later on, after we are serialized.
        self.bytecode = None

        if renpy.game.script.record_pycode:
            renpy.game.script.all_pycode.append(self)

        self.hash = None

    def get_hash(self):
        try:
            if self.hash is not None:
                return self.hash
        except:
            pass

        code = self.source
        if isinstance(code, renpy.python.ast.AST): #@UndefinedVariable
            code = renpy.python.ast.dump(code) #@UndefinedVariable
        
        self.hash = chr(renpy.bytecode_version) + md5.md5(repr(self.location) + code.encode("utf-8")).digest()
        return self.hash
            
            
def chain_block(block, next): #@ReservedAssignment
    """
    This is called to chain together all of the nodes in a block. Node
    n is chained with node n+1, while the last node is chained with
    next.
    """

    if not block:
        return

    for a, b in zip(block, block[1:]):
        a.chain(b)

    block[-1].chain(next)


class Scry(object):
    """
    This is used to store information about the future, if we know it. Unlike
    predict, this tries to only get things we _know_ will happen.
    """

    # By default, all attributes are None.
    def __getattr__(self, name):
        return None

    def next(self): #@ReservedAssignment
        if self._next is None:
            return None
        else:
            return self._next.scry()

    
class Node(object):
    """
    A node in the abstract syntax tree of the program.

    @ivar name: The name of this node.

    @ivar filename: The filename where this node comes from.
    @ivar linenumber: The line number of the line on which this node is defined.
    """
    
    __slots__ = [
        'name',
        'filename',
        'linenumber',
        'next',
        ]

    # True if this node is translatable, false otherwise. (This can be set on
    # the class or the instance.)
    translatable = False

    # Called to set the state of a Node, when necessary.
    def __setstate__(self, state):
        for k, v in state[1].iteritems():
            try:           
                setattr(self, k, v)
            except AttributeError:
                pass
            
            
    def __init__(self, loc):
        """
        Initializes this Node object.

        @param loc: A (filename, physical line number) tuple giving the
        logical line on which this Node node starts.
        """

        self.filename, self.linenumber  = loc
        self.name = None
        self.next = None
        
    def diff_info(self):
        """
        Returns a tuple of diff info about ourself. This is used to
        compare Nodes to see if they should be considered the same node. The
        tuple returned must be hashable.
        """

        return ( id(self), )

    def get_children(self):
        """
        Returns a list of all of the nodes that are children of this
        node. (That is, all of the nodes in any block associated with
        this node.)
        """

        return [ ]

    def get_init(self):
        """
        Returns a node that should be run at init time (that is, before
        the normal start of the script.), or None if this node doesn't
        care to suggest one.

        (The only class that needs to override this is Init.)
        """

        return None
        
    def chain(self, next): #@ReservedAssignment
        """
        This is called with the Node node that should be followed after
        executing this node, and all nodes that this node
        executes. (For example, if this node is a block label, the
        next is the node that should be executed after all nodes in
        the block.)        
        """

        self.next = next

    def execute(self):
        """
        Causes this node to execute, and any action it entails to be
        performed. The node should call next_node with the node to 
        be executed after this one.
        """

        raise Exception("Node subclass forgot to define execute.")

    def early_execute(self):
        """
        Called when the module is loaded.
        """

    def predict(self):
        """
        This is called to predictively load images from this node.  It
        should cause renpy.display.predict.image and
        renpy.display.predict.screen to be called as necessary.
        """

        if self.next:
            return [ self.next ]
        else:
            return [ ]

    def scry(self):
        """
        Called to return an object with some general, user-definable information
        about the future.
        """

        rv = Scry()
        rv._next = self.next # W0201
        return rv

    def restructure(self, callback):
        """
        Called to restructure the AST.

        When this method is called, callback is called once for each child 
        block of the node. The block, a list, can be updated by the callback
        using slice assignment to the list.
        """
        
        # Does nothing for nodes that do not contain child blocks.
        return

    def get_code(self, dialogue_filter=None):
        """
        Returns the canonical form of the code corresponding to this statement.
        This only needs to be defined if the statement is translatable.
     
        `filter`
            If present, a filter that should be applied to human-readable
            text in the statement.
        """

        raise Exception("Not Implemented")

def say_menu_with(expression, callback):
    """
    This handles the with clause of a say or menu statement.
    """

    if expression is not None:
        what = renpy.python.py_eval(expression)
    elif renpy.store.default_transition and renpy.game.preferences.transitions == 2:
        what = renpy.store.default_transition
    else:
        return

    if not what:
        return

    if renpy.game.preferences.transitions:
        # renpy.game.interface.set_transition(what)
        callback(what)
        
class Say(Node):

    __slots__ = [
        'who',
        'who_fast',
        'what',
        'with_',
        'interact',
        'attributes',
        ]

    def diff_info(self):
        return (Say, self.who, self.what)

    def __setstate__(self, state):
        self.attributes = None
        self.interact = True
        Node.__setstate__(self, state)

    def __init__(self, loc, who, what, with_, interact=True, attributes=None):

        super(Say, self).__init__(loc)

        if who is not None:
            self.who = who.strip()

            if re.match(r'[a-zA-Z_]\w*$', self.who):
                self.who_fast = True
            else:
                self.who_fast = False
        else:
            self.who = None 
            self.who_fast = False
            
        self.what = what
        self.with_ = with_
        self.interact = interact

        # A tuple of attributes that are applied to the character that's
        # speaking, or None to disable this behavior.
        self.attributes = attributes

    def get_code(self, dialogue_filter=None):
        rv = [ ]
        
        if self.who:
            rv.append(self.who)
            
        if self.attributes is not None:
            rv.extend(self.attributes)
            
        what = self.what
        if dialogue_filter is not None:
            what = dialogue_filter(what)
            
        rv.append(renpy.translation.encode_say_string(what))

        if not self.interact:
            rv.append("nointeract")

        if self.with_:           
            rv.append("with")
            rv.append(self.with_)

        return " ".join(rv)

    def execute(self):
        
        next_node(self.next)

        try:

            renpy.exports.say_attributes = self.attributes
            
            if self.who is not None:
                if self.who_fast:
                    who = getattr(renpy.store, self.who, None)
                    if who is None:
                        raise Exception("Sayer '%s' is not defined." % self.who.encode("utf-8"))
                else:
                    who = renpy.python.py_eval(self.who)
            else:
                who = None
    
            if not (
                (who is None) or
                callable(who) or
                isinstance(who, basestring) ):
    
                raise Exception("Sayer %s is not a function or string." % self.who.encode("utf-8"))
                
            what = self.what
            if renpy.config.say_menu_text_filter:
                what = renpy.config.say_menu_text_filter(what) # E1102
                
            if getattr(who, "record_say", True):
                renpy.store._last_say_who = self.who
                renpy.store._last_say_what = what
    
            say_menu_with(self.with_, renpy.game.interface.set_transition)
            renpy.exports.say(who, what, interact=self.interact)

        finally:
            renpy.exports.say_attributes = None
            
    
    def predict(self):

        old_attributes = renpy.exports.say_attributes

        try:
            
            renpy.exports.say_attributes = self.attributes
            
            if self.who is not None:
                if self.who_fast:
                    who = getattr(renpy.store, self.who)
                else:
                    who = renpy.python.py_eval(self.who)
            else:
                who = None
    
            def predict_with(trans):
                renpy.display.predict.displayable(trans(old_widget=None, new_widget=None))
    
            say_menu_with(self.with_, predict_with)
    
            what = self.what
            if renpy.config.say_menu_text_filter:
                what = renpy.config.say_menu_text_filter(what)
    
            renpy.exports.predict_say(who, what)

        finally:
            renpy.exports.say_attributes = old_attributes
        
        return [ self.next ]

    def scry(self):
        rv = Node.scry(self)

        if self.who is not None:
            if self.who_fast:
                who = getattr(renpy.store, self.who)
            else:
                who = renpy.python.py_eval(self.who)
        else:
            who = None

        if self.interact:            
            renpy.exports.scry_say(who, rv)
        else:
            rv.interacts = False
            
        return rv
    
# Copy the descriptor.
setattr(Say, "with", Say.with_) # E1101

class Init(Node):

    __slots__ = [
        'block',
        'priority',
        ]

    def __init__(self, loc, block, priority):
        super(Init, self).__init__(loc)

        self.block = block
        self.priority = priority


    def get_children(self):
        return self.block

    def get_init(self):
        return self.priority, self.block[0]

    # We handle chaining specially. We want to chain together the nodes in
    # the block, but we want that chain to end in None, and we also want
    # this node to just continue on to the next node in normal execution.
    def chain(self, next): #@ReservedAssignment
        self.next = next

        chain_block(self.block, None)

    def execute(self):
        next_node(self.next)
    
    def restructure(self, callback):
        callback(self.block)
    
def apply_arguments(params, args, kwargs):
    """
    Applies arguments to parameters to update scope.
    
    `scope`
        A dict.
        
    `params`
        The parameters object.
        
    `args`, `kwargs`
        Positional and keyword arguments.
    """
    
    values = { }
    rv = { }
    
    if args is None:
        args = ()

    if kwargs is None:
        kwargs = { }

    if params is None:
        if args or kwargs:
            raise Exception("Arguments supplied, but parameter list not present")
        else:
            return rv
    
    for name, value in zip(params.positional, args):
        if name in values:
            raise Exception("Parameter %s has two values." % name)

        values[name] = value

    extrapos = tuple(args[len(params.positional):])

    for name, value in kwargs.iteritems():
        if name in values:
            raise Exception("Parameter %s has two values." % name)

        values[name] = value

    for name, default in params.parameters:

        if name not in values:
            if default is None:
                raise Exception("Required parameter %s has no value." % name)
            else:
                rv[name] = renpy.python.py_eval(default)
        
        else:
            rv[name] = values[name]
            del values[name]
                
    # Now, values has the left-over keyword arguments, and extrapos
    # has the left-over positional arguments.

    if params.extrapos:
        rv[params.extrapos] = extrapos
    elif extrapos:
        raise Exception("Too many arguments in call (expected %d, got %d)." % (len(params.positional), len(args)))

    if params.extrakw:
        rv[params.extrakw] = values
    else:
        if values:
            raise Exception("Unknown keyword arguments: %s" % ( ", ".join(values.keys())))

    return rv
    
class Label(Node):

    __slots__ = [
        'name',
        'parameters',
        'block',
        'hide',
        ]

    def __setstate__(self, state):
        self.parameters = None
        self.hide = False
        Node.__setstate__(self, state)
    
    def __init__(self, loc, name, block, parameters, hide=False):
        """
        Constructs a new Label node.

        @param name: The name of this label.
        @param block: A (potentially empty) list of nodes making up the
        block associated with this label.
        """

        super(Label, self).__init__(loc)

        self.name = name
        self.block = block
        self.parameters = parameters
        self.hide = hide

    def diff_info(self):
        return (Label, self.name)

    def get_children(self):
        return self.block 

    def chain(self, next): #@ReservedAssignment

        if self.block:
            self.next = self.block[0]
            chain_block(self.block, next)
        else:
            self.next = next
            
    def execute(self):
        next_node(self.next)

        renpy.game.context().mark_seen()
        
        values = apply_arguments(self.parameters, renpy.store._args, renpy.store._kwargs)

        for k, v in values.iteritems():
            renpy.exports.dynamic(k)
            setattr(renpy.store, k, v)

        renpy.store._args = None
        renpy.store._kwargs = None

        if renpy.config.label_callback:
            renpy.config.label_callback(self.name, renpy.game.context().last_abnormal)

    def restructure(self, callback):
        callback(self.block)


class Python(Node):

    __slots__ = [
        'hide',
        'code',
        'store',
        ]

    def __setstate__(self, state):
        self.store = "store"
        super(Python, self).__setstate__(state)

    def __init__(self, loc, python_code, hide=False, store="store"):
        """
        @param code: A PyCode object.

        @param hide: If True, the code will be executed with its
        own local dictionary.
        """
        
        super(Python, self).__init__(loc)

        self.hide = hide
        self.code = PyCode(python_code, loc=loc, mode='exec')
        self.store = store

    def diff_info(self):
        return (Python, self.code.source)

    def early_execute(self):
        renpy.python.create_store(self.store)

    def execute(self):
        next_node(self.next)
        
        try:        
            renpy.python.py_exec_bytecode(self.code.bytecode, self.hide, store=self.store)
        finally:
            
            if not renpy.game.context().init_phase:
                for i in renpy.config.python_callbacks:
                    i()

    def scry(self):
        rv = Node.scry(self)
        rv.interacts = True
        return rv

class EarlyPython(Node):

    __slots__ = [
        'hide',
        'code',
        'store',
        ]

    def __setstate__(self, state):
        self.store = "store"
        super(EarlyPython, self).__setstate__(state)

    def __init__(self, loc, python_code, hide=False, store="store"):
        """
        @param code: A PyCode object.

        @param hide: If True, the code will be executed with its
        own local dictionary.
        """
        
        super(EarlyPython, self).__init__(loc)

        self.hide = hide
        self.code = PyCode(python_code, loc=loc, mode='exec')
        self.store = store

    def diff_info(self):
        return (EarlyPython, self.code.source)

    def execute(self):
        next_node(self.next)

    def early_execute(self):
        renpy.python.create_store(self.store)
        
        if self.code.bytecode:
            renpy.python.py_exec_bytecode(self.code.bytecode, self.hide, store=self.store)

class Image(Node):

    __slots__ = [
        'imgname',
        'code',
        'atl',
        ]

    def __init__(self, loc, name, expr=None, atl=None):
        """
        @param name: The name of the image being defined.

        @param expr: An expression yielding a Displayable that is
        assigned to the image.
        """

        super(Image, self).__init__(loc)
        
        self.imgname = name

        if expr:
            self.code = PyCode(expr, loc=loc, mode='eval')
            self.atl = None
        else:
            self.code = None
            self.atl = atl
            
    def diff_info(self): 
        return (Image, tuple(self.imgname))

    def execute(self):

        # Note: We should always check that self.code is None before
        # accessing self.atl, as self.atl may not always exist.

        next_node(self.next)

        if self.code is not None:
            img = renpy.python.py_eval_bytecode(self.code.bytecode)
        else:
            img = renpy.display.motion.ATLTransform(self.atl)

        renpy.exports.image(self.imgname, img)


    
class Transform(Node):

    __slots__ = [

        # The name of the transform.
        'varname',

        # The block of ATL associated with the transform.
        'atl',

        # The parameters associated with the transform, if any.
        'parameters',        
        ]

    default_parameters = ParameterInfo([ ], [ ], None, None)
    
    def __init__(self, loc, name, atl=None, parameters=default_parameters):

        super(Transform, self).__init__(loc)
        
        self.varname = name
        self.atl = atl
        self.parameters = parameters
        
    def diff_info(self): 
        return (Transform, self.varname)

    def execute(self):

        next_node(self.next)

        parameters = getattr(self, "parameters", None)

        if parameters is None:
            parameters = Transform.default_parameters

        trans = renpy.display.motion.ATLTransform(self.atl, parameters=parameters)
        renpy.dump.transforms.append((self.varname, self.filename, self.linenumber))
        setattr(renpy.store, self.varname, trans)

    
def predict_imspec(imspec, scene=False, atl=None):
    """
    Call this to use the given callback to predict the image named
    in imspec.
    """

    if len(imspec) == 7:
        name, expression, tag, _at_list, layer, _zorder, _behind = imspec

    elif len(imspec) == 6:
        name, expression, tag, _at_list, layer, _zorder = imspec

    elif len(imspec) == 3:
        name, _at_list, layer = imspec
        
        
    if expression:
        try:
            img = renpy.python.py_eval(expression)
            img = renpy.easy.displayable(img)
        except:
            return

    else:
        img = renpy.display.image.images.get(name, None)
        if img is None:
            return

    full_name = name
    if tag:
        full_name = (tag,) + full_name[1:]

    if scene:
        renpy.game.context().images.predict_scene(layer)
        
    renpy.game.context().images.predict_show(tag or name, layer)
    
    if atl is not None:
        try:
            img = renpy.display.motion.ATLTransform(atl, child=img)
        except:
            import traceback
            traceback.print_exc()
        
    renpy.display.predict.displayable(img)


    
def show_imspec(imspec, atl=None):

    if len(imspec) == 7:
        name, expression, tag, at_list, layer, zorder, behind = imspec
        
    elif len(imspec) == 6:
        name, expression, tag, at_list, layer, zorder = imspec
        behind = [ ]
        
    elif len(imspec) == 3:
        name, at_list, layer = imspec
        expression = None
        tag = None
        zorder = None
        behind = [ ]
        
    if zorder is not None:
        zorder = renpy.python.py_eval(zorder)
    else:
        zorder = 0

    if expression is not None:
        expression = renpy.python.py_eval(expression)
        expression = renpy.easy.displayable(expression)

    at_list = [ renpy.python.py_eval(i) for i in at_list ]

    renpy.config.show(name,
                      at_list=at_list,
                      layer=layer,
                      what=expression,
                      zorder=zorder,
                      tag=tag,
                      behind=behind,
                      atl=atl)

class Show(Node):

    __slots__ = [
        'imspec',
        'atl',
        ]

    def __init__(self, loc, imspec, atl=None):
        """
        @param imspec: A triple consisting of an image name (itself a
        tuple of strings), a list of at expressions, and a layer.
        """

        super(Show, self).__init__(loc)

        self.imspec = imspec
        self.atl = atl
        
    def diff_info(self): 
        return (Show, tuple(self.imspec[0]))

    def execute(self):
        next_node(self.next)
        
        show_imspec(self.imspec, atl=getattr(self, "atl", None))

    def predict(self):
        predict_imspec(self.imspec, atl=getattr(self, "atl", None))
        return [ self.next ]
        

class Scene(Node):

    __slots__ = [
        'imspec',
        'layer',
        'atl',
        ]

    def __init__(self, loc, imgspec, layer, atl=None):
        """
        @param imspec: A triple consisting of an image name (itself a
        tuple of strings), a list of at expressions, and a layer, or
        None to not have this scene statement also display an image.
        """

        super(Scene, self).__init__(loc)

        self.imspec = imgspec
        self.layer = layer
        self.atl = atl

    def diff_info(self): 

        if self.imspec:
            data = tuple(self.imspec[0])
        else:
            data = None

        return (Scene, data)

    def execute(self):

        next_node(self.next)

        renpy.config.scene(self.layer)

        if self.imspec:
            show_imspec(self.imspec, atl=getattr(self, "atl", None))
        
    def predict(self):
        
        if self.imspec:
            predict_imspec(self.imspec, atl=getattr(self, "atl", None), scene=True)

        return [ self.next ]

    
class Hide(Node):

    __slots__ = [
        'imspec',
        ]

    def __init__(self, loc, imgspec):
        """
        @param imspec: A triple consisting of an image name (itself a
        tuple of strings), a list of at expressions, and a list of
        with expressions.
        """

        super(Hide, self).__init__(loc)

        self.imspec = imgspec

    def diff_info(self): 
        return (Hide, tuple(self.imspec[0]))

    def predict(self):

        if len(self.imspec) == 3:
            name, _at_list, layer = self.imspec
            tag = None
            _expression = None
            _zorder = None
            _behind = None
        elif len(self.imspec) == 6:
            name, _expression, tag, _at_list, layer, _zorder = self.imspec
            _behind = None
        elif len(self.imspec) == 7:
            name, _expression, tag, _at_list, layer, _zorder, _behind = self.imspec


        if tag is None:
            tag = name[0]
            
        renpy.game.context().images.predict_hide(tag, layer)

        return [ self.next ]
        
    def execute(self):

        next_node(self.next)

        if len(self.imspec) == 3:
            name, _at_list, layer = self.imspec
            _expression = None
            tag = None
            _zorder = 0
        elif len(self.imspec) == 6:
            name, _expression, tag, _at_list, layer, _zorder = self.imspec
        elif len(self.imspec) == 7:
            name, _expression, tag, _at_list, layer, _zorder, _behind = self.imspec
            
        renpy.config.hide(tag or name, layer)

    
class With(Node):

    __slots__ = [
        'expr',
        'paired',
        ]

    def __setstate__(self, state):
        self.paired = None
        Node.__setstate__(self, state)
    
    def __init__(self, loc, expr, paired=None):
        """
        @param expr: An expression giving a transition or None.
        """

        super(With, self).__init__(loc)
        self.expr = expr
        self.paired = paired

    def diff_info(self):
        return (With, self.expr)
        
    def execute(self):

        next_node(self.next)

        trans = renpy.python.py_eval(self.expr)

        if self.paired is not None:
            paired = renpy.python.py_eval(self.paired)
        else:
            paired = None 

        renpy.exports.with_statement(trans, paired)

    def predict(self):

        try:
            trans = renpy.python.py_eval(self.expr)

            if trans:
                renpy.display.predict.displayable(trans(old_widget=None, new_widget=None))

        except:
            pass

                
        return [ self.next ]
    
        
class Call(Node):

    __slots__ = [
        'label',
        'arguments',
        'expression',
        ]

    def __setstate__(self, state):
        self.arguments = None
        Node.__setstate__(self, state)
    
    def __init__(self, loc, label, expression, arguments):

        super(Call, self).__init__(loc)
        self.label = label
        self.expression = expression
        self.arguments = arguments

    def diff_info(self):
        return (Call, self.label, self.expression)

    def execute(self):

        label = self.label
        if self.expression:
            label = renpy.python.py_eval(label)

        rv = renpy.game.context().call(label, return_site=self.next.name)
        next_node(rv)
        renpy.game.context().abnormal = True

        if self.arguments:

            args = [ ]
            kwargs = renpy.python.RevertableDict()

            for name, expr in self.arguments.arguments:

                value = renpy.python.py_eval(expr)

                if name is None:
                    args.append(value)
                else:
                    if name in kwargs:
                        raise Exception("The argument named %s appears twice." % name)

                    kwargs[name] = value

            if self.arguments.extrapos:
                args.extend(renpy.python.py_eval(self.arguments.extrapos))

            if self.arguments.extrakw:
                for name, value in renpy.python.py_eval(self.arguments.extrakw).iteritems():
                    if name in kwargs:
                        raise Exception("The argument named %s appears twice." % name)

                    kwargs[name] = value


            renpy.store._args = tuple(args)
            renpy.store._kwargs = kwargs
                            
    def predict(self):
        if self.expression:
            return [ ]
        else:
            return [ renpy.game.script.lookup(self.label) ]

    def scry(self):
        rv = Node.scry(self)
        rv._next = None
        return rv


class Return(Node):

    __slots__ = [ 'expression']

    def __setstate__(self, state):
        self.expression = None
        Node.__setstate__(self, state)
    
    def __init__(self, loc, expression):
        super(Return, self).__init__(loc)
        self.expression = expression
        
    def diff_info(self):
        return (Return, )

    # We don't care what the next node is.
    def chain(self, next): #@ReservedAssignment
        self.next = None
        return

    def execute(self):

        if self.expression:
            renpy.store._return = renpy.python.py_eval(self.expression)
        else:
            renpy.store._return = None

        renpy.game.context().pop_dynamic()
            
        next_node(renpy.game.context().lookup_return(pop=True))

    def predict(self):
        site = renpy.game.context().lookup_return(pop=False)
        if site:
            return [ site ]
        else:
            return [ ]

    def scry(self):
        rv = Node.scry(self)
        rv._next = None
        return rv

    
class Menu(Node):

    __slots__ = [
        'items',
        'set',
        'with_',
        ]

    def __init__(self, loc, items, set, with_): #@ReservedAssignment
        super(Menu, self).__init__(loc)

        self.items = items
        self.set = set
        self.with_ = with_

    def diff_info(self):
        return (Menu,)

    def get_children(self):
        rv = [ ]

        for _label, _condition, block in self.items:
            if block:
                rv.extend(block)

        return rv

    # Blocks of statements in a choice continue after the menu.
    def chain(self, next): #@ReservedAssignment

        self.next = next

        for (_label, _condition, block) in self.items:
            if block:
                chain_block(block, next)

    def execute(self):

        next_node(self.next)

        choices = [ ]
        narration = [ ]        
        
        for i, (label, condition, block) in enumerate(self.items):

            if renpy.config.say_menu_text_filter:
                label = renpy.config.say_menu_text_filter(label)

            if block is None:
                if renpy.config.narrator_menu and label:
                    narration.append(label)
                else:
                    choices.append((label, condition, None))
            else:
                choices.append((label, condition, i))
                next_node(block[0])

        if narration:
            renpy.exports.say(None, "\n".join(narration), interact=False)
                
        say_menu_with(self.with_, renpy.game.interface.set_transition)
        choice = renpy.exports.menu(choices, self.set)

        if choice is not None:
            next_node(self.items[choice][2][0])
        else:
            next_node(self.next)
        

    def predict(self):
        rv = [ ]

        def predict_with(trans):
            renpy.display.predict.displayable(trans(old_widget=None, new_widget=None))

        say_menu_with(self.with_, predict_with)

        renpy.store.predict_menu()
        
        for _label, _condition, block in self.items:
            if block:
                rv.append(block[0])

        return rv

    def scry(self):
        rv = Node.scry(self)
        rv._next = None
        rv.interacts = True
        return rv
    
    def restructure(self, callback):
        for _label, _condition, block in self.items:
            if block is not None:
                callback(block)
    
setattr(Menu, "with", Menu.with_) # E1101


# Goto is considered harmful. So we decided to name it "jump"
# instead. 
class Jump(Node):

    __slots__ = [
        'target',
        'expression',
        ]

    def  __init__(self, loc, target, expression):
        super(Jump, self).__init__(loc)

        self.target = target
        self.expression = expression

    def diff_info(self):
        return (Jump, self.target, self.expression)

    # We don't care what our next node is.
    def chain(self, next): #@ReservedAssignment
        self.next = None
        return

    def execute(self):

        target = self.target
        if self.expression:
            target = renpy.python.py_eval(target)

        rv = renpy.game.script.lookup(target)
        renpy.game.context().abnormal = True

        next_node(rv)

    def predict(self):

        if self.expression:
            return [ ]
        else:
            return [ renpy.game.script.lookup(self.target) ]

    def scry(self):
        rv = Node.scry(self)
        if self.expression:
            rv._next = None
        else:
            rv._next = renpy.game.script.lookup(self.target)
            
        return rv


# GNDN
class Pass(Node):

    __slots__ = [ ]

    def diff_info(self):
        return (Pass,)

    def execute(self):
        next_node(self.next)


class While(Node):

    __slots__ = [
        'condition',
        'block',
        ]

    def __init__(self, loc, condition, block):
        super(While, self).__init__(loc)

        self.condition = condition
        self.block = block

    def diff_info(self):
        return (While, self.condition)

    def get_children(self):
        return self.block

    def chain(self, next): #@ReservedAssignment
        self.next = next
        chain_block(self.block, self)

    def execute(self):

        next_node(self.next)

        if renpy.python.py_eval(self.condition):
            next_node(self.block[0])

    def predict(self):
        return [ self.block[0], self.next ]
        
    def scry(self):
        rv = Node.scry(self)
        rv._next = None
        return rv

    def restructure(self, callback):
        callback(self.block)

class If(Node):

    __slots__ = [ 'entries' ]

    def __init__(self, loc, entries):
        """
        @param entries: A list of (condition, block) tuples.
        """

        super(If, self).__init__(loc)

        self.entries = entries

    def diff_info(self):
        return (If,)

    def get_children(self):
        rv = [ ]

        for _condition, block in self.entries:
            rv.extend(block)

        return rv

    def chain(self, next): #@ReservedAssignment
        self.next = next

        for _condition, block in self.entries:
            chain_block(block, next)

    def execute(self):

        next_node(self.next)

        for condition, block in self.entries:
            if renpy.python.py_eval(condition):
                next_node(block[0])
                return

    def predict(self):

        return [ block[0] for _condition, block in self.entries ] + \
               [ self.next ]

    def scry(self):
        rv = Node.scry(self)
        rv._next = None
        return rv

    def restructure(self, callback):
        for _condition, block in self.entries:
            callback(block)

class UserStatement(Node):

    __slots__ = [ 
        'line', 
        'parsed', 
        'block', 
        'translatable' ]

    def __setstate__(self, state):
        self.block = [ ]
        self.translatable = False
        Node.__setstate__(self, state)
     
    def __init__(self, loc, line, block):

        super(UserStatement, self).__init__(loc)
        self.line = line
        self.block = block
        self.parsed = None

        # Do not store the parse quite yet.
        _parse_info = renpy.statements.parse(self, self.line, self.block)
        
    def diff_info(self):
        return (UserStatement, self.line)

    def execute(self):
        next_node(self.get_next())
        
        self.call("execute")

    def predict(self):
        self.call("predict")            
        return [ self.get_next() ]
    
    def call(self, method, *args, **kwargs):
        
        parsed = self.parsed        
        if parsed is None:
            parsed = renpy.statements.parse(self, self.line, self.block)
            self.parsed = parsed

        renpy.statements.call(method, parsed, *args, **kwargs)

    def get_next(self):
        rv = self.call("next")
        if rv is not None:
            return renpy.game.script.lookup(rv)
        else:
            return self.next
        
    def scry(self):
        rv = Node.scry(self)
        rv._next = self.get_next()
        self.call("scry", rv)
        return rv
     
    def get_code(self, dialogue_filter=None):
        return self.line
                            
            
class Define(Node):

    __slots__ = [
        'varname',
        'code',
        ]

    def __init__(self, loc, name, expr):
        """
        @param name: The name of the image being defined.

        @param expr: An expression yielding a Displayable that is
        assigned to the image.
        """

        super(Define, self).__init__(loc)
        
        self.varname = name
        self.code = PyCode(expr, loc=loc, mode='eval')
            
    def diff_info(self): 
        return (Define, tuple(self.varname))

    def execute(self):

        next_node(self.next)

        value = renpy.python.py_eval_bytecode(self.code.bytecode)
        renpy.dump.definitions.append((self.varname, self.filename, self.linenumber))
        setattr(renpy.store, self.varname, value)    


class Screen(Node):

    __slots__ = [
        'screen',
        ]

    def __init__(self, loc, screen):
        """
        @param name: The name of the image being defined.

        @param expr: An expression yielding a Displayable that is
        assigned to the image.
        """

        super(Screen, self).__init__(loc)
        
        self.screen = screen
            
    def diff_info(self): 
        return (Screen, self.screen.name)

    def execute(self):
        next_node(self.next)
        self.screen.define()
        renpy.dump.screens.append((self.screen.name, self.filename, self.linenumber))


################################################################################
# Translations
################################################################################

class Translate(Node):
    """
    A translation block, produced either by explicit translation statements 
    or implicit translation blocs.
    
    If language is None, when executed this transfers control to the translate 
    statement in the current language, if any, and otherwise runs the block.
    If language is not None, causes an error to occur if control reaches this 
    statement.

    When control normally leaves a translate statement, in any language, it 
    goes to the end of the translate statement in the None language.
    """
    
    __slots__ = [
        "identifier",
        "language",
        "block",
        ]
    
    def __init__(self, loc, identifier, language, block):
        super(Translate, self).__init__(loc)
        
        self.identifier = identifier
        self.language = language
        self.block = block
        
    def diff_info(self):
        return (Translate, self.identifier, self.language)

    def chain(self, next): #@ReservedAssignment
        self.next = next
        chain_block(self.block, next)

    def execute(self):
        if self.language is not None:
            next_node(self.next)
            raise Exception("Translation nodes cannot be run directly.")
        trans_node = renpy.game.script.translator.lookup_translate(self.identifier)

        trans_hash = trans_pack.get(self.identifier)
        if trans_hash is not None:
            trans_node.what = renpy.game.script.translator.lookup_translate(trans_hash).what
        next_node(trans_node)
        renpy.game.context().translate_identifier = self.identifier

    def predict(self):
        node = renpy.game.script.translator.lookup_translate(self.identifier)
        return [ node ]
    
    def scry(self):
        rv = Scry()
        rv._next = renpy.game.script.translator.lookup_translate(self.identifier)
        return rv
    
    def get_children(self):
        return self.block
    
    def restructure(self, callback):
        return callback(self.block)
    
    
class EndTranslate(Node):
    """
    A node added implicitly after each translate block. It's responsible for
    resetting the translation identifier.
    """
    
    def __init__(self, loc):
        super(EndTranslate, self).__init__(loc)
        
    def diff_info(self):
        return (EndTranslate,)
    
    def execute(self):
        next_node(self.next)
        renpy.game.context().translate_identifier = None
        

class TranslateString(Node):
    """
    A node used for translated strings.
    """
    
    __slots__ = [ 
        "language",
        "old",
        "new"
        ]
    
    def __init__(self, loc, language, old, new):
        super(TranslateString, self).__init__(loc)
        self.language = language
        self.old = old
        self.new = new
        
    def diff_info(self):
        return (TranslateString,)
    
    def execute(self):
        next_node(self.next)
        renpy.translation.add_string_translation(self.language, self.old, self.new)

class TranslatePython(Node):

    __slots__ = [
        'language',
        'code',
        ]

    def __init__(self, loc, language, python_code):
        """
        @param code: A PyCode object.

        @param hide: If True, the code will be executed with its
        own local dictionary.
        """
        
        super(TranslatePython, self).__init__(loc)

        self.language = language
        self.code = PyCode(python_code, loc=loc, mode='exec')

    def diff_info(self):
        return (TranslatePython, self.code.source)

    def execute(self):
        next_node(self.next)

    # def early_execute(self):
    #    renpy.python.create_store(self.store)
    #    renpy.python.py_exec_bytecode(self.code.bytecode, self.hide, store=self.store)

trans_pack = {
'tavern_22f150f9':'tavern_e77e5690',
'tavern_221a59f2':'tavern_469f1ccf',
'tavern_394f6375':'tavern_e82772ab',
'tavern_aa99ef38':'tavern_3bc3b8ec',
'tavern_1d0515b5':'tavern_d12316e4',
'tavern_926bd8d3':'tavern_5d4590fb',
'tavern_f36d80be':'tavern_67d22610',
'tavern_b2590317':'tavern_c03893c5',
'tavern_cea112f9':'tavern_8ce304ea',
'tavern_d85ecd5a':'tavern_d8c5ae9f',
'tavern_c947879b':'tavern_5c7ce541',
'tavern_4ba3063a':'tavern_9a096ff8',
'tavern_59476e7c':'tavern_1d0f229b',
'tavern_53ecd8f9':'tavern_ed4ebb85',
'tavern_4eaa45a4':'tavern_4a2efc05',
'tavern_7471d480':'tavern_aaa8fe16',
'tavern_69e86a08':'tavern_29929af2',
'tavern_b434ae38':'tavern_0797fdfa',
'tavern_6787ee51':'tavern_1cd9b03a',
'tavern_5ba2406b':'tavern_e429ebca',
'tavern_2ce0ccff':'tavern_99f3e559',
'tavern_0cab0455':'tavern_08db29ae',
'tavern_545d70e5':'tavern_5df566d0',
'tavern_06c50e52':'tavern_c0819ade',
'tavern_49519bac':'tavern_d2eab446',
'tavern_9228fc23':'tavern_e7c9bf3e',
'tavern_241c1b9c':'tavern_72de23e6',
'tavern_06c50e52_1':'tavern_c0819ade',
'tavern_49519bac_1':'tavern_d2eab446',
'tavern_ef6ee8dc_1':'tavern_ef6ee8dc',
'tavern_c19f56ec':'tavern_1d0f229b',
'tavern_24040922':'tavern_be247608',
'tavern_dd1cbc40':'tavern_eeba060e',
'tavern_ef222a17':'tavern_a2569e93',
'tavern_5759224b_1':'tavern_1d0f229b',
'tavern_26675226':'tavern_aa08f931',
'tavern_0a8054ea':'tavern_4b3af5b5',
'tavern_45f8db30':'tavern_89cf6be6',
'tavern_8060c0e4':'tavern_b253b62d',
'tavern_f120cfb5':'tavern_da208820',
'tavern_97ab7735':'tavern_2545d535',
'tavern_7bb3ad57':'tavern_e20ba10d',
'tavern_28d39592':'tavern_3e3a5b73',
'tavern_0e17da48':'tavern_29b961b0',
'tavern_626f6bc9':'tavern_fde80f97',
'tavern_d3241b97':'tavern_331b0560',
'tavern_9bee5f2b':'tavern_c35859e0',
'tavern_a07af3a8':'tavern_6c9f9d72',
'tavern_a45f03bb':'tavern_953597cc',
'tavern_e6f5345d':'tavern_8740222d',
'tavern_45133d55':'tavern_7bc0e504',
'tavern_58488733':'tavern_9a87507b',
'tavern_0828a3ed':'tavern_d5d6f002',
'tavern_04ac220d':'tavern_9d4c0068',
'tavern_2582086d':'tavern_f0b0ded4',
'tavern_3928694e':'tavern_dc699e0f',
'tavern_baf57c62':'tavern_12a03e69',
'tavern_3962188b':'tavern_0d188deb',
'tavern_981827be':'tavern_7f8232b9',
'tavern_83e3d02a':'tavern_1de3b759',
'tavern_26d80b64':'tavern_dac648c1',
'tavern_6ef0969d':'tavern_723422fb',
'tavern_7acdc8da':'tavern_2e483423',
'tavern_b86dde57':'tavern_b0e41e65',
'tavern_ebd981be':'tavern_447d87ec',
'tavern_84873222':'tavern_645c5267',
'tavern_0cf28fbf':'tavern_9166b553',
'tavern_8efe602e':'tavern_b480a0f0',
'tavern_fca13c6c':'tavern_6579fa8e',
'tavern_ed4895d7':'tavern_db0f9cc1',
'tavern_e4ea8412':'tavern_605378d7',
'tavern_f4f6d415':'tavern_d4b278a4',
'tavern_2267cc10':'tavern_46302c95',
'tavern_e8be2437':'tavern_27a4eae0',
'tavern_edb11b2f':'tavern_b0dce4dd',
'tavern_898a84db':'tavern_6ca56979',
'tavern_18e2f5e8':'tavern_d67b09e0',
'tavern_c9f700d1':'tavern_7727168b',
'tavern_aa828daf':'tavern_b5112249',
'tavern_14081226':'tavern_e618822e',
'tavern_4ea6a45c':'tavern_64a68575',
'tavern_e274afaf':'tavern_981ef57e',
'tavern_05c59a06':'tavern_4b9cc2ad',
'tavern_50574cca':'tavern_b55c1930',
'tavern_876cb5fa':'tavern_0ebd9a19',
'tavern_074c42f1':'tavern_d65af790',
'tavern_5c3f9838':'tavern_76f7e6cc',
'tavern_e4c92c1f':'tavern_080516c2',
'tavern_a8255ee1':'tavern_2c024dde',
'tavern_bf14981c':'tavern_acb9e20a',
'tavern_c1ee6be7':'tavern_4861b5ba',
'tavern_52b63e33':'tavern_b74a97f6',
'tavern_ea00ef20':'tavern_91bbf53e',
'tavern_6a95f93b':'tavern_52e02597',
'tavern_d057e919':'tavern_d210846d',
'tavern_f7de4ce7':'tavern_6a95147e',
'tavern_a3b89365':'tavern_a7c81ec5',
'tavern_97693445':'tavern_4e95bcf1',
'tavern_c2dbbe95':'tavern_a5f83eb8',
'tavern_46863928':'tavern_30c508bb',
'tavern_3958f114':'tavern_78abf2d1',
'tavern_c23f5b3e':'tavern_c2f8e6ac',
'tavern_3356656f':'tavern_6f128bd8',
'tavern_673b32d5':'tavern_143c0e72',
'tavern_ad61671a':'tavern_aa2938d0',
'tavern_60e056b0':'tavern_580d86a9',
'tavern_69055fe6':'tavern_a6294ee0',
'tavern_1108fcf1':'tavern_ca0969de',
'tavern_d83cb147':'tavern_4dca7acb',
'tavern_bd33942f':'tavern_5f3c9b2a',
'tavern_a3a9c4f3':'tavern_1ca97378',
'tavern_1cbb762a':'tavern_dfee9831',
'tavern_7a435c75':'tavern_55086c4d',
'tavern_7ea89b85':'tavern_fcf85b07',
'tavern_7b18a85d':'tavern_4700e9af',
'tavern_33ba095d':'tavern_343ef6ce',
'tavern_f135bdb9':'tavern_c1671209',
'tavern_ec4306ba':'tavern_bbe5ce42',
'tavern_a07cff1e':'tavern_a07cff1e',
'tavern_c8477fe6':'tavern_38f62f9e',
'tavern_b4dc30fd':'tavern_def59c17',
'tavern_64b73259':'tavern_26e5e029',
'tavern_46b9b8a9':'tavern_a4dc2cac',
'tavern_7b87ce76':'tavern_0721b0af',
'tavern_c01a735b':'tavern_f4558aed',
'tavern_b8af60bc':'tavern_db61a2a9',
'tavern_7051043c':'tavern_9e8a1a45',
'tavern_44c5b6bb':'tavern_af53ba1b',
'tavern_27de5b0a':'tavern_495e0675',
'tavern_363c106e':'tavern_43840575',
'tavern_6671a70b':'tavern_0d1b9d12',
'tavern_ebf69626':'tavern_a9af0d22',
'tavern_7d134c19':'tavern_3d98db75',
'tavern_47177ca9':'tavern_e3690673',
'tavern_8b355acc':'tavern_428e6712',
'tavern_28d39592_1':'tavern_3e3a5b73',
'tavern_19585e4c_1':'tavern_19585e4c',
'tavern_7a62ee27':'tavern_e32fb505',
'tavern_879bc341':'tavern_12e46f32',
'tavern_5bca1c0b':'tavern_70d1b214',
'tavern_438acda0':'tavern_790d6ccd',
'tavern_f9312b74':'tavern_84adc924',
'tavern_057251c2':'tavern_0f061810',
'tavern_f008c9f4':'tavern_a0417b6e',
'tavern_8921f78a':'tavern_cf207957',
'tavern_73d33c63':'tavern_1d9532ee',
'tavern_a3d7f939':'tavern_17ebd354',
'tavern_d6a2aa7a':'tavern_20f2c7f4',
'tavern_c2f42428':'tavern_bf909216',
'tavern_dc890ba8':'tavern_9e5be174',
'tavern_e3f346b6':'tavern_a15e564a',
'tavern_45d8dee2':'tavern_90763c73',
'tavern_58d65461':'tavern_f8f0fe91',
'tavern_a63445b1':'tavern_62e076bd',
'tavern_76282ec8':'tavern_1b1a6202',
'tavern_b8dc429e':'tavern_e0587e0f',
'tavern_fd7e25c3':'tavern_949df118',
'tavern_b30874cd':'tavern_32730b90',
'tavern_eff33c56':'tavern_324f7c3b',
'tavern_4fe09f51':'tavern_8a1b8d0a',
'tavern_5e8803f3_1':'tavern_5e8803f3',
'tavern_142648aa':'tavern_13ca2e46',
'tavern_980cd3e6':'tavern_46492036',
'tavern_84e70d7f_1':'tavern_84e70d7f',
'tavern_c2e49e23':'tavern_7bc9aea9',
'tavern_d057e919_1':'tavern_d210846d',
'tavern_b3c0b2d0':'tavern_c1c51927',
'tavern_89a614fc':'tavern_232ab382',
'tavern_b8a3b69d':'tavern_6a055093',
'tavern_1393bb1d':'tavern_31342fbb',
'tavern_fcf36930':'tavern_c0bfb636',
'tavern_fc1432cd':'tavern_0b97bf89',
'tavern_61b1529e':'tavern_80baffbc',
'tavern_3f41b169':'tavern_4da27b95',
'tavern_64901997':'tavern_28c6d676',
'tavern_645abaed':'tavern_6c38b58c',
'tavern_e3095cfc':'tavern_a9d53ac5',
'tavern_04d2025d':'tavern_846d4036',
'tavern_a5270452':'tavern_cd89e3a2',
'tavern_c4e25739':'tavern_098d6c31',
'tavern_2bd57f6d':'tavern_85dde2c5',
'tavern_013b1dcb':'tavern_52a456ca',
'tavern_5f69f0de':'tavern_9f93687d',
'tavern_d19032de':'tavern_a3302b52',
'tavern_6aa569f6':'tavern_6a39e807',
'tavern_838ba84f':'tavern_c60c704a',
'tavern_096a3eae':'tavern_fd94a507',
'tavern_3f70e178':'tavern_02fe3e8c',
'tavern_c08fa0df':'tavern_a5203818',
'tavern_ad2bab26':'tavern_57436ccb',
'tavern_83073e78':'tavern_52a18849',
'tavern_6b67c6bd':'tavern_2f3c24a4',
'tavern_393b23d6':'tavern_335e28f1',
'tavern_5e9e1845':'tavern_38949b1a',
'tavern_ef1f043c':'tavern_21461bcc',
'tavern_ce3e2df8':'tavern_bde8432a',
'tavern_312cb2d7':'tavern_5b0e2d61',
'tavern_0c14f669':'tavern_a0b8a9dd',
'tavern_158721f2':'tavern_5d2cc7b6',
'tavern_f22e06ae':'tavern_1b5d15dc',
'tavern_8080d96d':'tavern_a9cf2720',
'tavern_244a80cf':'tavern_ab16d446',
'tavern_5e22e662':'tavern_be5225a5',
'tavern_e3bfade7':'tavern_dc8d317d',
'tavern_22ec4a0d':'tavern_04b7f2df',
'tavern_826d7060':'tavern_6de72909',
'tavern_ba39cea3':'tavern_a0889ae3',
'tavern_8ca394fb':'tavern_ab8d8aea',
'tavern_c8e063ce':'tavern_5e34fe91',
'tavern_7c0a2de9':'tavern_971d14cc',
'tavern_2e1a8690':'tavern_405a14b6',
'tavern_ded4758c':'tavern_abcde031',
'tavern_ea4e0078':'tavern_314a3093',
'tavern_d7ad4c9f':'tavern_b5fd3ffa',
'tavern_f431c67c':'tavern_3553b1fc',
'tavern_6a583b7e':'tavern_e60eae74',
'tavern_f45ee091':'tavern_53889c06',
'tavern_b3961989':'tavern_6b4c005a',
'tavern_1c96111c':'tavern_7f3820ef',
'tavern_56d0bf88':'tavern_a6ceb14c',
'tavern_dcd27970':'tavern_e08e62d0',
'tavern_06cf8bc2':'tavern_9dc82732',
'tavern_513e9379':'tavern_7d0df821',
'tavern_213333e8':'tavern_1a5120dd',
'tavern_04f606b7':'tavern_239523ab',
'tavern_9117a22f':'tavern_4d27dcbf',
'tavern_adb0fdf2':'tavern_cfc65219',
'tavern_be597d27':'tavern_e7e0c287',
'tavern_4edc1e70':'tavern_a8a9d31b',
'tavern_f3cb00b2':'tavern_ae70333d',
'tavern_67b061ff':'tavern_e32160ac',
'tavern_549af769':'tavern_cbdd0ac8',
'tavern_d556ee6e':'tavern_683c4540',
'tavern_8c23f2ce':'tavern_87be18ea',
'tavern_f364e1a6':'tavern_07b45151',
'tavern_5b4ff2a8':'tavern_54feda8c',
'tavern_088949c4':'tavern_746066d5',
'tavern_fe9fed8a':'tavern_fc1558d0',
'tavern_df379ca5':'tavern_8684d290',
'tavern_1c2df7cd':'tavern_0f2c2997',
'tavern_11a59bd5':'tavern_8d2151f0',
'tavern_f35a31eb':'tavern_388ace82',
'tavern_534c0bff':'tavern_e01be1fa',
'tavern_baacbf27':'tavern_5bf1cd47',
'tavern_2a43fffc':'tavern_55582458',
'tavern_34178f7f':'tavern_41207752',
'tavern_f8d4fbb9':'tavern_2f9a238e',
'tavern_7ad6164b':'tavern_df75528b',
'tavern_4bb48e0d':'tavern_4df4a551',
'tavern_ac9a9195':'tavern_48ae7de4',
'tavern_d2f6e903':'tavern_9a7d349b',
'tavern_b362995d':'tavern_b8ed0eb1',
'tavern_b924a9a2':'tavern_7557e5ae',
'tavern_60277e86':'tavern_b3e2b055',
'tavern_7c22117d':'tavern_f5c27519',
'tavern_4ca9ec4a':'tavern_6668bde0',
'tavern_69d1c533':'tavern_5f1cf8fb',
'tavern_7f395022':'tavern_041c2a59',
'tavern_6c95ea59':'tavern_47a1473b',
'tavern_fea02c65':'tavern_b04dd6d3',
'tavern_2e72c2a3':'tavern_10444560',
'tavern_776c9e43':'tavern_1628b452',
'tavern_36c5c511':'tavern_af625aea',
'tavern_f5dcf15f':'tavern_feb980bd',
'tavern_de5ec3e0':'tavern_85ce469d',
'tavern_6e9ecf78':'tavern_627bc68a',
'tavern_cbf4dcf7':'tavern_c2d2607b',
'tavern_cf45c3b3':'tavern_adf7bc90',
'tavern_44095ebd':'tavern_f77831b6',
'tavern_cbf4dcf7_1':'tavern_c2d2607b',
'tavern_f8bb3a61':'tavern_41a25a87',
'tavern_5813acfb':'tavern_298da77b',
'tavern_24ce9788':'tavern_b86628f4',
'tavern_785ae1be':'tavern_4e55bf11',
'tavern_87d0a102':'tavern_c0c903a1',
'tavern_e112e8df':'tavern_f9e3fe8d',
'tavern_63bfbc8c':'tavern_ede9abc3',
'tavern_dec8c95c':'tavern_cd79d3fc',
'tavern_2768ce95':'tavern_4d78a8e8',
'tavern_55162b35':'tavern_58c54ba0',
'tavern_3ef27580':'tavern_f011e000',
'tavern_68fda223':'tavern_174edbc4',
'tavern_0fe0757a':'tavern_eacccb23',
'tavern_5e5f10a7':'tavern_ec73cbc8',
'tavern_dab678c3':'tavern_5ef5e292',
'tavern_db0e194a':'tavern_ebea8db7',
'tavern_8bf15a95':'tavern_1ab6ea86',
'tavern_7848387d':'tavern_0e2ffaa1',
'tavern_cb6b3758':'tavern_15802e6d',
'tavern_9e2e2400':'tavern_1481e0d7',
'tavern_5a0fa5a8':'tavern_38898cc3',
'tavern_43e99205':'tavern_fe20ac61',
'tavern_39d1bd7e':'tavern_78f7b3c4',
'tavern_dc3353aa':'tavern_553dc9dc',
'tavern_7f993b35':'tavern_30c651d0',
'tavern_8691546f':'tavern_120618c4',
'tavern_d5a19f35':'tavern_74c3847d',
'tavern_7143b080':'tavern_85b956e0',
'tavern_7cf6db77':'tavern_7bd536a2',
'tavern_30fa785b':'tavern_ec38ca84',
'tavern_64530a86':'tavern_932bfa7c',
'tavern_cb4fe2f7':'tavern_71652100',
'tavern_da737949':'tavern_e64d11b6',
'tavern_c0ef573e':'tavern_321964fb',
'tavern_218bdcf5':'tavern_b1c8ea10',

'master_51bd8144':'master_641d9402',
'master_84ad6ac1':'master_3ef3611d',
'master_060caff3':'master_cb8a9fc0',
'master_d0d2f219':'master_d71c9876',
'master_55d9b4e0':'master_ec4d7d2e',
'master_b05a6a96':'master_9010f6bd',
'master_4ad4457b':'master_7a09f60c',
'master_7149c7d5':'master_0a892a08',
'master_9a0bc925':'master_d1abfe23',
'master_e42dc5b8':'master_b9b70aa9',
'master_05baf13b':'master_c559b0ad',
'master_462c04ba':'master_aae2bf9b',
'master_2e2bad1d':'master_69ff36d6',
'master_4ff31046':'master_16a53cd1',
'master_19b81594':'master_a473680d',
'master_26ad09fc':'master_d1dee28b',
'master_28aae9ee':'master_da8c836a',
'master_0370f329':'master_3f3c50d0',
'master_3e53c429':'master_a0669e0f',
'master_cebc5b22':'master_aaf4b59f',
'master_4ac350e6':'master_2611ca87',
'master_c97b1e0c':'master_74dab7dd',
'master_b9ee8661':'master_0952024b',
'master_403cfd4b':'master_e3d73ea5',
'master_9fed9c52':'master_4c3fb3e7',
'master_6457df03':'master_3bddefc1',
'master_23c78d90':'master_5eb6a4fc',
'master_17d92fb9':'master_e6d13f52',
'master_0a7549a4':'master_6697319c',
'master_e85018e5':'master_71d86ca0',
'master_f37b5142':'master_af5bfd74',
'master_19bf8167':'master_e5d9adba',
'master_34f1d028':'master_bf657f54',
'master_d850ea7a':'master_c8c290c1',
'master_1785262e':'master_66f052df',
'master_eebe35a6':'master_df10c824',
'master_07b0d3b8':'master_39d443ad',
'master_49836d6c':'master_ddded1c8',
'master_d849ff54':'master_7a4049a7',
'master_8c18176d':'master_f6d4adde',
'master_7f5c9265':'master_4a7e1bc7',
'master_56388587':'master_039cf416',
'master_edf169f4':'master_5e91c963',
'master_afd9d1ce':'master_849f7abc',
'master_baf18b0f':'master_44ceda4a',
'master_bd1e35e4':'master_9fe2acb0',
'master_20ea0ede':'master_080da9a9',
'master_d5c3d62b':'master_138a10e0',
'master_0c5c7d70':'master_c076a371',
'master_df9fe6b2':'master_a851a29c',
'master_db59c570':'master_2037da7b',
'master_c6cd65e9':'master_e9bac627',
'master_48870d90':'master_aae2bf9b',
'master_74a3a9d7':'master_96ed8219',
'master_92514160':'master_63995de7',
'master_d1f45cab':'master_36115045',
'master_fc9850d6':'master_a581a2d8',
'master_2077fb18':'master_fe18625b',
'master_faaa5a9e':'master_d1dee28b',
'master_53d5d74e':'master_9656fc55',
'master_f70d295d':'master_c6bb8e32',
'master_03f4bc40':'master_ab90a354',
'master_68a00352':'master_cfe52f0d',
'master_16325433':'master_13fb0fe0',
'master_1164cf95':'master_04097436',
'master_ab164dfe':'master_220e0b62',
'master_2dc5f493':'master_88cc3183',
'master_efecc878':'master_111cbc1e',
'master_88cd6b69':'master_c51bf528',
'master_98997d1f':'master_5505bd1c',
'master_7b21d6b5':'master_0bfc42ca',
'master_06ecc64e':'master_b1838ef7',
'master_d166326f':'master_52993f88',
'master_95e99bd8':'master_5dcc4f88',
'master_43f7ee58':'master_ef56f1ee',
'master_1d72ad9d':'master_2cad6a73',
'master_1fd9e6d0':'master_c8cfb39d',
'master_e23bc9fe':'master_bc10c1c6',
'master_ea2b1b22':'master_56c8b2cc',
'master_bea874c0':'master_3c602018',
'master_709b52b0':'master_3dd7a843',
'master_eef42235':'master_0b359fca',
'master_89719de2':'master_393b31c8',
'master_5f754229':'master_bcd83063',
'master_8b1fc12b':'master_031a7c6c',
'master_3c5bdc4e':'master_6203b213',
'master_17dc0163':'master_971d1d18',
'master_4ab9a584':'master_fcad7ad6',
'master_2bdabeff':'master_965f04dd',
'master_14d8375b':'master_e5ffc46e',
'master_44697f18':'master_ebfb8391',
'master_2048b5da':'master_ae83bd90',
'master_583801de':'master_26d0634f',
'master_1dd00877':'master_f95ca7b7',
'master_c5ca8490':'master_efc0844e',
'master_994d9238':'master_b6bf6ad5',
'master_68c175dc':'master_1077f493',
'master_e5ab60c8':'master_7c94e59c',
'master_5c212bbb':'master_69d8049d',
'master_c67bc99b':'master_b7921c40',
'master_b0aeb2e0':'master_d9a9b4b9',
'master_e77aad9d':'master_bbceed0a',
'master_36b08b05':'master_e8859bbd',
'master_1aa76be1':'master_5645b9cf',
'master_02c6900f':'master_6435cecf',
'master_5e1d1dce':'master_caa9fd05',
'master_74927f2d':'master_989d2a81',
'master_face95da':'master_9aa6b824',
'master_4499bc61':'master_f671c0b6',
'master_7adbc4a1':'master_0c238cba',
'master_c0a0e735':'master_f00a47ec',
'master_15cf9de2':'master_f4c32a3d',
'master_5ec59a34':'master_f08320c1',
'master_dd6aaf25':'master_8e17f802',
'master_8f95efa6':'master_2b452439',
'master_bd19fcc2':'master_ea3e9e89',
'master_abeca780':'master_cc20a6c6',
'master_178abf56':'master_b1b14683',
'master_50f27c2b':'master_3dd89278',
'master_cfedc705':'master_fe77a39f',
'master_1f497592':'master_902ade4b',
'master_23ccd4b0':'master_2a7778cb',
'master_fe5e6f35':'master_6140c1d6',
'master_9bf7cbc4':'master_bbcd09fa',
'master_5badd4c7':'master_906f7371',
'master_e0f9781e':'master_3b72e572',
'master_5ce0055b':'master_3b033360',
'master_469cc62f':'master_d4b1c660',
'master_8fea28b2':'master_6d747e7d',
'master_c6c3a832':'master_9620ee0b',
'master_5d22a6c7':'master_28bb962c',
'master_d18bd121':'master_5d4b69d9',
'master_b02d38d9':'master_584ed8c1',
'master_823865f1':'master_72b46866',
'master_8e2e5d3b':'master_0cec7924',
'master_0dc77b8b':'master_d6ab700d',
'master_887cd84b':'master_e3181bc0',
'master_5595a04d':'master_c64f1d4a',
'master_7845eb29':'master_d1275487',
'master_1c74638d':'master_ceb438b2',
'master_864bc372':'master_13167206',
'master_f70d295d_1':'master_c6bb8e32',
'master_20437394':'master_4addc894',
'master_0204cabc':'master_6f5ef006',
'master_5745f9f6':'master_b0b575bf',
'master_4034536e':'master_b2945308',
'master_f58b901a':'master_23b38534',
'master_9fe0487f':'master_b4b8130c',
'master_dfa5c59a':'master_0d053f32',
'master_0d592a9c':'master_3655dbae',
'master_140264db':'master_9f266f87',
'master_5a58e563':'master_1de7e05b',
'master_dceb47b1':'master_8fd2a004',
'master_2025c636':'master_5759224b',
'master_18b41e40':'master_71c6fc35',
'master_84cbd10f':'master_738e8d5c',
'master_cf93a30e':'master_596d6939',
'master_8034d7b0':'master_32a5b1e6',
'master_c67ae914':'master_354a6da7',
'master_dec1d2f9':'master_960c8f79',
'master_9e73d219':'master_e738a7e2',
'master_39fe5987':'master_d7700b82',
'master_ecec3cb1':'master_e07166f8',
'master_d45e99bd':'master_8986c71f',
'master_a7a4c586':'master_b2fb2424',
'master_65e1d384':'master_3bbde764',
'master_8fce1bf4':'master_2c985244',
'master_89f1210e':'master_23a5307b',
'master_8728565f':'master_017af6dd',
'master_640db76d':'master_070562de',
'master_756d6c5f':'master_80168172',
'master_780df9fd':'master_c4a64e7b',
'master_449de92b':'master_7f5aa3af',
'master_1e18c91f':'master_024f8eeb',
'master_307618e4':'master_f2e44d9b',
'master_cfe6032c':'master_9d34eb6b',
'master_3b57ed8c':'master_58519b06',
'master_9098415a':'master_e92f0515',
'master_e3a835ca':'master_6d4182ff',
'master_80baa595':'master_381792df',
'master_4195495c':'master_d99e9347',
'master_a9791d1a':'master_dae54eed',
'master_0d5660b5':'master_299e0b18',
'master_474fa296':'master_29d51d1f',
'master_1c77ea26':'master_6bcb5059',
'master_e2a1340c':'master_d64bbc3f',
'master_417772f6':'master_5f8d6366',
'master_92a00390':'master_119703f7',
'master_1d5b0972':'master_c2e7e318',
'master_a7cee208':'master_8753876a',
'master_9d4db103':'master_a6388d97',
'master_cbbe39f2':'master_54331333',
'master_c24a559a':'master_5ccf7418',
'master_2a126725':'master_c8f35c66',
'master_23eeb930':'master_61b09739',
'master_15057598':'master_948e8d0d',
'master_ffc52744':'master_6ec10a56',
'master_5d3506fb':'master_c8421a00',
'master_affc7d23':'master_a8717dca',
'master_66549ef4':'master_ff9a6299',
'master_ad74fed6':'master_1115e6c3',
'master_6d28f256':'master_366d61a8',
'master_ad64d9f8':'master_3fa6168d',
'master_e1c71ea5':'master_b1ff7b78',
'master_3250557d':'master_8a25528a',
'master_c339dc9d':'master_517acfd6',
'master_efe95775':'master_71698602',
'master_5009aaa7':'master_a5180ac9',
'master_3b0ee16b':'master_005342fa',
'master_36d5c331':'master_f9704505',
'master_50286fdb':'master_078aab4d',
'master_910b3b38':'master_86472814',
'master_6fcc6e99':'master_5fe3b23b',
'master_4c401691':'master_819aed90',
'master_a0c8bf9d':'master_e51eeeba',
'master_6ff5ca0e':'master_d81e08c2',
'master_73a56460':'master_08f484ff',
'master_0b68d4d8':'master_2de9508a',
'master_172b9c03':'master_fc0db33e',
'master_c63e0c9f':'master_524c39b0',
'master_aa66b097':'master_29a0046b',
'master_d41ada44':'master_3aa623d0',
'master_72c47aa1':'master_9ad121b7',
'master_05fb28da':'master_93ef21ea',
'master_21788ff1':'master_c5689da0',
'master_e4eb8f39':'master_47f74697',
'master_a1d787c7':'master_c6b3dea2',
'master_20ce7c90':'master_21dfaa76',
'master_bd631ebd':'master_6e50d775',
'master_5e76ee88':'master_7131efaf',
'master_185cc982':'master_49a21705',
'master_4823cd72':'master_be505a8a',
'master_72b30f26':'master_90d1f0d1',
'master_c83d82da':'master_0c82b4cb',
'master_a326ab02':'master_5e358bec',
'master_5b55ece5':'master_31b3b62d',
'master_0e59f102':'master_34e9b07a',
'master_64453569':'master_66d3c5c4',
'master_4eb0ca38':'master_1a6ab803',
'master_9b3059d4':'master_5759224b',
'master_d209009d':'master_1179b781',
'master_ad64d9f8_1':'master_3fa6168d',
'master_ae8297af':'master_f4e5d930',
'master_90abce02':'master_7b467c3d',
'master_784c2b83':'master_e3d803df',
'master_991bd352':'master_bb28473e',
'master_3120277b':'master_2a85f4e1',
'master_a626db1a':'master_26ce2f61',
'master_ef095f2c':'master_c5926fdd',
'master_e3763fb3':'master_a5ef47a3',
'master_7c389aec':'master_f91d7911',
'master_f6ede4fa':'master_eb76db08',
'master_ab33b12e':'master_f9eef611',
'master_731e1d69':'master_df6ba13f',
'master_0e0462ec':'master_905600ac',
'master_688e9056':'master_e3d57e53',
'master_245c4b6e':'master_be196219',
'master_426f1de6':'master_a0524332',
'master_4ff19440':'master_6ad57322',
'master_3098c628':'master_9ab28b36',
'master_9cd04167':'master_93a1d5d2',
'master_63d605c2':'master_88e64e70',
'master_48b1f075':'master_c1ce16d2',
'master_caf51715':'master_1a85cf15',
'master_8c2ad8e2':'master_3867b57a',
'master_04a67f40':'master_016ed028',
'master_c1b76ea9':'master_a33436dd',
'master_2067a244':'master_bf0017c5',
'master_607164b0':'master_58779b38',
'master_3b526331':'master_51df7375',
'master_719bca52':'master_6083a406',
'master_af1ea59c':'master_f2b8b3fe',
'master_eda993df':'master_e7e05301',
'master_311d3868':'master_2f031946',
'master_0e011892':'master_94d953b5',
'master_53f9b55e':'master_dcb27935',
'master_23ba6fcb':'master_65278c62',
'master_235acd33':'master_00701953',
'master_ff5ab626':'master_9a5a5e21',
'master_96a8ed45':'master_39b69b1a',
'master_ad1a5113':'master_fb643ce2',
'master_9f4b7dd3':'master_95ee1355',
'master_b6409694':'master_b539f96b',
'master_e2fe2857':'master_3f3499f4',
'master_3eac7b3b':'master_37838d77',
'master_7bd308a4':'master_38092708',
'master_84782ca9':'master_dad23d88',
'battle_astarte_7407f3b2':'battle_astarte_050ae070',

'shop_d7e5e1bf':'shop_d7e5e1bf',
'shop_14b0c3ca':'shop_94d5d80f',
'shop_62f21d13':'shop_dd32ee42',
'shop_853ce50d':'shop_9d5d0112',
'shop_c71dffca':'shop_440a5727',
'shop_d7e5e1bf':'shop_3a04a741',
'shop_e05a7e37':'shop_efe4a7dd',
'shop_52a25b6b':'shop_0f9196ea',
'shop_1cee14f3':'shop_0ff6918a',
'shop_456d17dc':'shop_9d20f57e',
'shop_6eef9557':'shop_25ee3035',
'shop_5a2adcdf':'shop_0fb5b7e5',
'shop_28edffdc':'shop_52721e8f',
'shop_d6d0007b':'shop_455e1017',
'shop_a5f0a492':'shop_407ac58f',
'shop_64d8c570':'shop_3ef8cbad',
'shop_a68a5265':'shop_a68a5265',
'shop_361fd2c5':'shop_ba885f8a',
'shop_41e0c603':'shop_e2aad9ce',
'shop_262eb2cc':'shop_8a275609',
'shop_99753bab':'shop_34557bdc',
'shop_5fde192f':'shop_607362fa',
'shop_03956a23':'shop_24317ad7',
'shop_012207f1':'shop_903ca5dc',
'shop_9dc31446':'shop_0e2d80f1',
'shop_b075b773':'shop_8043c012',
'shop_043a32a5':'shop_398fdc65',
'shop_0411ab6f':'shop_9dfc6e1e',
'shop_3a7c62df':'shop_6d747e7d',
'shop_ad2bab26':'shop_57436ccb',
'shop_4f24c7b8':'shop_ea4b9796',
'shop_850fc58b':'shop_4394a4f0',

'prologue_703d49ae':'prologue_2ea1dd0b',
'prologue_fa8233a3':'prologue_77e15bbb',
'prologue_701f3cea':'prologue_d460a2c5',
'prologue_58e91960':'prologue_ac2d00a7',
'prologue_d1c686a9':'prologue_0fc727e5',
'prologue_ecaf8ac4':'prologue_a3159f14',
'prologue_c4d635a2':'prologue_c90387b0',
'prologue_c9071f26':'prologue_cecfad5f',
'prologue_770caa17':'prologue_15956e8d',
'prologue_3dc0ab59':'prologue_a17232cd',
'prologue_cfbdfb86':'prologue_2c03bb41',
'prologue_7d98c46d':'prologue_889f1b98',
'prologue_08c5acc4':'prologue_442ae875',
'prologue_ca85f789':'prologue_fb7d93ef',
'prologue_240d13cb':'prologue_3f09b1a4',
'prologue_206efded':'prologue_6afb262d',
'prologue_a789088c':'prologue_8abd6fd5',
'prologue_345ed223':'prologue_29fa49a1',
'prologue_d6f0ccbf':'prologue_0156ccb7',
'prologue_554bb3e6':'prologue_c20d46db',
'prologue_c51a601d':'prologue_433fc7b2',
'prologue_9433c345':'prologue_774a8062',
'prologue_a7387df8':'prologue_fcc6c593',
'prologue_6dc6052a':'prologue_ecec3b87',
'prologue_e2c0a7bb':'prologue_2dbae1ef',
'prologue_93ba7b99':'prologue_b3ea0ab9',
'prologue_d67489a3':'prologue_53c53e8a',
'prologue_36c4406f':'prologue_55290174',
'prologue_e29e9759':'prologue_442f767f',
'prologue_f1fa0f26':'prologue_8f7400a6',
'prologue_ce5c72dc':'prologue_6eedc4b3',
'prologue_3b6d303d':'prologue_c5719118',
'prologue_6e52f466':'prologue_7f80eb17',
'prologue_49e72c97':'prologue_df4dad1b',
'prologue_b04a4147':'prologue_4aca28fb',
'prologue_d1d0bacb':'prologue_75efb991',
'prologue_b19665c0':'prologue_693f25da',
'prologue_6f929fef':'prologue_27d263fd',
'levia_prologue_2cc1cca0':'levia_prologue_33b39633',
'prologue2_88bf361a':'prologue2_839e8035',
'prologue2_51f6d418':'prologue2_2babfa56',
'prologue2_f018cec2':'prologue2_11b28030',
'prologue2_b549c46f':'prologue2_258c1084',
'prologue2_ad6d6f6e':'prologue2_09663e9b',
'prologue2_4218e3fb':'prologue2_9644f037',
'prologue2_67b86cf6':'prologue2_a95b75e3',
'prologue2_852e278c':'prologue2_d1356914',
'prologue2_66d83ac3':'prologue2_fd32fb3f',
'prologue2_bb556823':'prologue2_e5d9aea5',
'prologue2_b3f8eddb':'prologue2_22fdb721',
'prologue2_cfbba5ab':'prologue2_33637d8d',
'prologue2_1c2a1de0':'prologue2_6c47e28a',
'prologue2_9978b407':'prologue2_980780a9',
'prologue2_a55ed946':'prologue2_6cedf57f',
'prologue2_2d972054':'prologue2_b1bd4f39',
'prologue2_e3ad2bb1':'prologue2_f2e4fbef',
'prologue2_5b5a12e1':'prologue2_5e7d368e',
'prologue2_8eeea713':'prologue2_0bbff7f6',
'prologue2_cf913728':'prologue2_1e1948c8',
'prologue2_f126f9ec':'prologue2_9853ae51',
'prologue2_125b7fe0':'prologue2_fc70fc35',
'prologue2_d4cceca3':'prologue2_d8d5577a',
'prologue2_62cbc051':'prologue2_3329b039',
'prologue2_bda661a8':'prologue2_3306c592',
'prologue2_f124f9b6':'prologue2_ff534617',
'prologue2_de879647':'prologue2_33e2bf64',
'prologue2_cd624600':'prologue2_fa377c22',
'prologue2_5b9eb7a1':'prologue2_2abd9321',
'prologue2_b66c9a49':'prologue2_8ccca08a',
'prologue2_5e996169':'prologue2_44562871',
'prologue2_2e58d98c':'prologue2_b1d720bd',
'prologue2_5500c53a':'prologue2_cd71dc3d',
'prologue2_5c38ff00':'prologue2_a539ce7e',
'prologue2_397f3899':'prologue2_0c832307',
'prologue2_6235d0c2':'prologue2_5ed84d48',
'prologue2_df68b55c':'prologue2_7d8a511e',
'prologue2_29d92970':'prologue2_3032cba5',
'prologue2_27bc78f3':'prologue2_3a8d441a',
'prologue2_84dc644f':'prologue2_e2f8114b',
'prologue2_47704738':'prologue2_62e72762',
'prologue2_01ec0656':'prologue2_c7179650',
'prologue2_90d30fa9':'prologue2_76ff2b89',
'prologue2_9df2cb9f':'prologue2_cd468284',
'prologue3_5f80fe59':'prologue3_6a8f8ad8',
'prologue3_94db84b2':'prologue3_797ce8a6',
'prologue3_0f1d25f6':'prologue3_0ece502a',
'prologue3_90020848':'prologue3_9200782d',
'prologue3_d0efb21a':'prologue3_906141dd',
'prologue3_9ed75443':'prologue3_6fd0797b',
'prologue3_938c1a1d':'prologue3_24845205',
'prologue3_01cec4c7':'prologue3_483ca739',
'prologue3_15931f1a':'prologue3_72408f65',
'prologue3_697c4dc5':'prologue3_9fee13e3',
'prologue3_fe6db3f9':'prologue3_e633176f',
'prologue3_7ca6ef49':'prologue3_bdfc5c22',
'prologue3_18f4d119':'prologue3_29d6d3b7',
'prologue3_d925f567':'prologue3_c3b04d2e',
'prologue3_bf568a19':'prologue3_f2d4be70',
'prologue3_bab2c51c':'prologue3_7728b81f',
'prologue3_837a357b':'prologue3_837a357b',
'prologue3_0888e14e':'prologue3_f51bc9b0',
'prologue3_d48d7010':'prologue3_3572400b',
'prologue3_8a071038':'prologue3_c58ff943',
'prologue3_5170d22f':'prologue3_b8491321',
'prologue3_ad389e86':'prologue3_34ed09b0',
'prologue3_c4933900':'prologue3_d6ab147b',
'prologue3_6b344f28':'prologue3_af58f1b4',
'prologue3_cc83304f':'prologue3_ad2e2a19',
'prologue3_9705dd74':'prologue3_3b98b239',
'prologue3_5f91da88':'prologue3_b72e959f',
'prologue3_4c96abc6':'prologue3_a048dc40',
'prologue3_02bddcf8':'prologue3_3572400b',
'prologue3_d4449598':'prologue3_06ce0419',
'prologue3_74e06b26':'prologue3_bff76fd2',
'prologue3_a3e12529':'prologue3_76f5d496',
'prologue3_d1018e61':'prologue3_cf39188e',
'prologue3_ad623fcd':'prologue3_5eda1c68',
'prologue3_04d2c471':'prologue3_1e73e87a',
'prologue3_2ad1f214':'prologue3_4d549a97',
'prologue3_fedb653e':'prologue3_1cc2dba4',
'prologue3_d48d7010_1':'prologue3_3572400b',
'prologue3_177b0db3':'prologue3_370ae10b',
'prologue3_4cfad08d':'prologue3_4cfad08d',
'prologue3_cdc90234':'prologue3_6acb5548',
'prologue3_3a906a23':'prologue3_b4ac684a',
'prologue3_0a09d79a':'prologue3_d47cb837',
'prologue3_66f8b639':'prologue3_ef15f124',
'prologue3_ca34d555':'prologue3_2b1e6334',
'prologue3_2543147a':'prologue3_e980c19c',
'prologue3_b0778585':'prologue3_ad28e534',
'prologue3_ddba2236':'prologue3_a271f58b',
'prologue3_9870a609':'prologue3_9c3ca4fc',
'prologue3_b4b11748':'prologue3_3c944151',
'prologue3_2e5e5669':'prologue3_fea0c1fe',
'prologue3_812b008e':'prologue3_ec59dc65',
'prologue3_3fefbcbe':'prologue3_481c44ab',
'prologue3_8a723fa7':'prologue3_6c19e8ef',
'prologue3_c16f592e':'prologue3_92e3f776',
'prologue3_7d1f76dc':'prologue3_a82ee6f4',
'prologue3_cab5977c':'prologue3_66d9fbc0',
'prologue3_5cfe8967':'prologue3_8e1a701c',
'prologue3_4feba5a0':'prologue3_7d0a7b28',
'prologue3_547d5252':'prologue3_c4ef1967',
'prologue3_62c1c200':'prologue3_f61e9292',
'prologue3_1d28769b':'prologue3_b2645c0d',
'prologue3_f6541b38':'prologue3_888f2173',
'prologue3_a9f266d1':'prologue3_f747dc33',
'prologue3_6c74c2c7':'prologue3_1a62630e',
'prologue3_bd4025b6':'prologue3_36e3ce18',
'prologue3_87d8bc8c':'prologue3_05cf81fa',
'prologue3_307f9bab':'prologue3_5c85fc44',
'prologue3_66de433a':'prologue3_06ffa5a9',
'prologue3_64ce6565':'prologue3_bfbf8f4f',
'prologue3_747ffb28':'prologue3_5891a777',
'prologue3_9beddfaa':'prologue3_a95cb781',
'prologue3_4f9dddd1':'prologue3_56a10058',
'prologue3_505df529':'prologue3_64c5394e',
'prologue3_9e1b86a6':'prologue3_2f6f6e4b',
'prologue3_7af69f9a':'prologue3_e8482c0b',
'prologue3_f90d4be6':'prologue3_f5a882eb',
'prologue3_5cba2266':'prologue3_e31adcc1',
'prologue3_10a4eea9':'prologue3_2940d046',
'prologue3_64db64d1':'prologue3_fe240471',
'prologue3_8222f9a5':'prologue3_f31dcb2d',
'prologue3_fba04adc':'prologue3_9a8a1697',
'prologue3_ed9e07fa':'prologue3_2efe44e3',
'prologue3_92107265':'prologue3_1028ffaa',
'prologue3_047ca367':'prologue3_2e43366f',
'prologue3_82f41d5d':'prologue3_7b919af7',
'prologue3_6555cb0a':'prologue3_70c6941b',
'prologue3_c0aca781':'prologue3_858c5a74',
'prologue3_86be3d60':'prologue3_b614d465',
'prologue3_cb26999e':'prologue3_274bf71e',
'prologue3_19e73528':'prologue3_a46d3477',
'prologue3_90a87936':'prologue3_2daa93b5',
'prologue3_c4e4e885':'prologue3_bf470098',
'prologue3_ae97c4af':'prologue3_b83788be',
'prologue3_a2c245fd':'prologue3_353e4016',
'prologue3_85a8c272':'prologue3_bd7b0fab',
'prologue3_b958f206':'prologue3_811be5ff',
'prologue3_82b5fc06':'prologue3_cb7c0c7b',
'prologue3_9ae6eec7':'prologue3_4d522869',
'prologue3_ab3e2b05':'prologue3_09a56f8c',
'prologue3_c0313da0':'prologue3_21361ed7',
'prologue3_99f890aa':'prologue3_516a5371',
'prologue3_a645e6df':'prologue3_8d83f2b5',
'prologue3_78c309b2':'prologue3_450dfc7d',
'prologue3_525d1241':'prologue3_0202efac',
'prologue3_be550eff':'prologue3_56e65906',

'stage1_a_ed457846':'stage1_a_a6df53d9',
'stage1_a_0e2c3b95':'stage1_a_f8c349c7',
'stage1_a_8b36cde4':'stage1_a_6377ea21',
'stage1_a_c93bd9b9':'stage1_a_a28c9388',
'stage1_a_83d1c6bd':'stage1_a_6ff8f1b4',
'stage1_b_b4b73357':'stage1_b_b4b73357',
'stage1_b_615033f6':'stage1_b_f8c891f8',
'stage1_b_c5e70158':'stage1_b_2f2695a1',
'stage1_e_897fd919':'stage1_e_8e2df157',
'stage1_e_36059a13':'stage1_e_c7084e77',
'stage1_e_81ac9253':'stage1_e_20bacbb5',
'stage1_e_0faf6cb0':'stage1_e_c36cb0fa',
'stage1_e_fb6968bd':'stage1_e_ba7f12e3',
'stage1_e_c81fb502':'stage1_e_f94e3daa',
'stage1_g_6370cc5a':'stage1_g_0f2adf36',
'stage1_h_1ee72b87':'stage1_h_d8697442',
'stage1_h_2ba633bc':'stage1_h_80e55f61',
'stage1_h_6f82ce96':'stage1_h_791c95cf',
'stage1_h_984b0629':'stage1_h_2a7050ca',
'stage1_k_a4357bb3':'stage1_k_3f438196',
'stage1_k_ec8abac7':'stage1_k_8b5e2178',
'stage1_m_43744897':'stage1_m_b0ff997c',
'stage1_m_ec8abac7':'stage1_m_8b5e2178',
'battle_mouse_f21269fb':'battle_mouse_0ff65a91',
'battle_mouse_a75903bd':'battle_mouse_ff016411',
'battle_mouse_7a9b1b10':'battle_mouse_4fa9eedb',
'battle_mouse_e85a82ca':'battle_mouse_0b472f40',
'battle_mouse_1ff4c00d':'battle_mouse_5e66b16c',
'battle_mouse_e2d6298f':'battle_mouse_11dd930c',
'battle_mouse_59837bc5':'battle_mouse_216d6364',
'battle_mouse_2372732a':'battle_mouse_36008909',
'battle_slime_fa9ec197':'battle_slime_7d73a242',
'battle_slime_22879132':'battle_slime_6f3d2389',
'battle_slime_00fc19b7':'battle_slime_804218d6',
'battle_slime_6bd05202':'battle_slime_d42ba32e',

'stage2_b_5a422838':'stage2_b_2910dbd3',
'stage2_b_dacfb057':'stage2_b_af44366c',
'stage2_b_94a83a64':'stage2_b_542a42b7',
'stage2_b_4107e878':'stage2_b_42b88987',
'stage2_b_c364c10b':'stage2_b_5e7dd98c',
'stage2_b_3643579e':'stage2_b_64e7fd76',
'stage2_b_b97bdc3a':'stage2_b_1fe33935',
'stage2_b_e72f5e3a':'stage2_b_5fc4b0ed',
'stage2_b_9bc68d22':'stage2_b_ff4c96fe',
'stage2_b_b9cca88c':'stage2_b_f7f0e36e',
'stage2_b_1ec83ae4':'stage2_b_07ea55ce',
'stage2_b_4a269877':'stage2_b_371df359',
'stage2_c_854f2214':'stage2_c_50258d1c',
'stage2_c_402bb8b1':'stage2_c_7cd77834',
'stage2_c_0f04adf0':'stage2_c_0c98c56f',
'stage2_c_1770759b':'stage2_c_ca9641ee',
'stage2_c_eff22b7c':'stage2_c_e6fefff6',
'stage2_c_a5e59857':'stage2_c_cf48ca54',
'stage2_d_1770759b':'stage2_d_ca9641ee',
'stage2_d_ce6c4ea9':'stage2_d_23516b87',
'stage2_d_72f817a2':'stage2_d_b6457af3',
'stage2_d_c4f30668':'stage2_d_38449e43',
'stage2_e_6f2938e8':'stage2_e_00d2c0c4',
'stage2_e_ecfe2ef4':'stage2_e_5791df02',
'stage2_e_b103432b':'stage2_e_79c8117a',
'battle_sorcerer_c332931e':'battle_sorcerer_3017cc8d',
'battle_sorcerer_3f5398e7':'battle_sorcerer_94701d10',
'battle_sorcerer_df7c410b':'battle_sorcerer_4170eb05',
'battle_sorcerer_edbf048e':'battle_sorcerer_c06cad53',
'battle_sorcerer_4c50d2b4':'battle_sorcerer_2ad8c4e2',

'stage3_trap_bfeaa3ec':'stage3_trap_888f6367',
'stage3_trap_9a75a4f3':'stage3_trap_4dc6734a',
'stage3_trap_d5884926':'stage3_trap_49eafb0f',
'stage3_trap_0511565f':'stage3_trap_4acf8845',
'stage3_trap_7e74d7c9':'stage3_trap_de3de020',
'stage3_trap_3974b6e1':'stage3_trap_11720bfe',
'stage3_trap_53742a06':'stage3_trap_d307bed2',
'stage3_trap_ebd9f809':'stage3_trap_60bf0eb1',
'stage3_trap_aef04bef':'stage3_trap_0410518f',
'stage3_trap_16badf13':'stage3_trap_d83eeefb',
'stage3_trap_18839af4':'stage3_trap_ac55452e',
'stage3_trap_709fd3ad':'stage3_trap_cc1ba177',
'stage3_b_c583ba56':'stage3_b_cd8d4ced',
'stage3_c_c583ba56':'stage3_b_cd8d4ced',
'stage3_e_c583ba56':'stage3_b_cd8d4ced',
'stage3_f_c583ba56':'stage3_b_cd8d4ced',
'stage3_g_c583ba56':'stage3_b_cd8d4ced',
'stage3_h_c583ba56':'stage3_b_cd8d4ced',
'stage3_k_c583ba56':'stage3_b_cd8d4ced',
'stage3_m_c583ba56':'stage3_b_cd8d4ced',
'stage3_n_c583ba56':'stage3_b_cd8d4ced',
'stage3_o_c583ba56':'stage3_b_cd8d4ced',
'stage3_p_c583ba56':'stage3_b_cd8d4ced',
'stage3_s_3e3a226b':'stage3_s_7d3bee61',
'stage3_s_4476a8c6':'stage3_s_a6a85e93',
'stage3_u_b5cbec0b':'stage3_u_76e6218c',
'battle_succubus_0703d1e5':'battle_succubus_9ba8c918',
'battle_succubus_18876c85':'battle_succubus_28f916b8',
'battle_succubus_2b0a1c85':'battle_succubus_aeda1bdd',

'stage4_a_67732d27':'stage4_a_3fa8ecae',
'stage4_a_ec7de354':'stage4_a_a13755b6',
'stage4_a_cb02b803':'stage4_a_311d572d',
'stage4_g_d2acf9ec':'stage4_g_9d0ac173',
'stage4_g_890d1d95':'stage4_g_5de49da7',
'stage4_h_32cfdd0b':'stage4_h_8687e3de',
'stage4_k_cd3a6458':'stage4_k_e109e519',
'stage4_n_1a13cfb1':'stage4_n_9024274d',
'stage4_o_733fd6a1':'stage4_o_963a1691',
'stage4_o_4caa24a0':'stage4_o_2ecba4e8',
'stage4_o_c31a6c34':'stage4_o_a8aaeec2',
'stage4_o_9fc33251':'stage4_o_cd9f3f55',
'stage4_o_f9717cd1':'stage4_o_f5bbab62',
'stage4_o_448fc9d4':'stage4_o_11f0991c',
'stage4_o_4a1e8ba9':'stage4_o_ae448d68',
'stage4_o_9da14975':'stage4_o_6784b9da',
'stage4_o_89f6f874':'stage4_o_86205514',
'stage4_p_55beaeae':'stage4_p_eb5a6581',
'stage4_p_af9939a5':'stage4_p_2d168c19',
'stage4_p_e6dafb55':'stage4_p_5f32db55',
'stage4_q_760b53bd':'stage4_q_7e22a136',
'stage4_q_73e7c918':'stage4_q_aca2dd45',
'stage4_q_bf333b37':'stage4_q_a91e3e17',
'stage4_q_ea8febe2':'stage4_q_f2c127db',
'stage4_q_fb8bed79':'stage4_q_0e4027be',
'stage4_s_f3c35a76':'stage4_s_5e09c95f',
'stage4_s_4abc6b3d':'stage4_s_53b2b9cd',
'stage4_s_ef64e71e':'stage4_s_6029d882',
'stage4_s_5f80fe59':'stage4_s_6a8f8ad8',
'stage4_s_195855ee':'stage4_s_f696ae2b',
'stage4_s_036fd8d4':'stage4_s_46070767',
'stage4_s_c4a6fb1a':'stage4_s_97551ddc',
'stage4_s_60b3e322':'stage4_s_212b55c9',
'stage4_u_a35ca6cf':'stage4_u_d344c470',
'battle_mimic_1c87815c':'battle_mimic_4470d387',
'battle_mimic_a2fd4836':'battle_mimic_df11e60c',
'battle_darkelf_99b7d34a':'battle_darkelf_e01257e3',

'stage5_a_66ca3e86':'stage5_a_8f66c6cc',
'stage5_a_794783fc':'stage5_a_e5181876',
'stage5_a_5c52ef23':'stage5_a_b082d3dc',
'stage5_a_fe8209ae':'stage5_a_1d8a2d6d',
'stage5_a_6698098c':'stage5_a_12c67039',
'stage5_a_4b6d1d8b':'stage5_a_499d27df',
'stage5_a_403c38ef':'stage5_a_f7cb27fb',
'stage5_n_4110b645':'stage5_n_8c107140',
'stage5_n_4fae091b':'stage5_n_81894e4c',
'stage5_n_4472aaa7':'stage5_n_98e5bb00',
'stage5_n_a18d949c':'stage5_n_42702524',
'stage5_n_67ce183c':'stage5_n_fb974e68',
'stage5_n_ad64d9f8':'stage5_n_3fa6168d',
'stage5_n_61d55799':'stage5_n_09065628',
'stage5_n_80c2847b':'stage5_n_e0aa140a',
'stage5_n_bdb5c80e':'stage5_n_38f7c71f',
'stage5_n_00520d14':'stage5_n_9e501a80',
'stage5_n_87e63d68':'stage5_n_a7dbb3e5',
'stage5_n_0fc5daad':'stage5_n_f53b22b4',
'stage5_n_7676b03c':'stage5_n_3bc989ae',
'stage5_n_63716da6':'stage5_n_af9892c3',
'stage5_n_28cc6d9b':'stage5_n_f88af69a',
'stage5_n_a6043f5e':'stage5_n_63b8bb13',
'stage5_n_963ec309':'stage5_n_85a19de1',
'stage5_o_1bde46ba':'stage5_o_923c9daa',
'stage5_p_ae6bd207':'stage5_p_58d97939',
'stage5_q_31396142':'stage5_q_ec30da5d',
'stage5_q_711bb636':'stage5_q_42be3cf8',
'stage5_q_36ef6b15':'stage5_q_b466ac60',
'stage5_q_b8e7efda':'stage5_q_19ba0eb3',
'stage5_q_084e914e':'stage5_q_4ebad8a1',
'stage5_q_ac669f48':'stage5_q_b7f473db',
'stage5_q_eba56f9f':'stage5_q_ec45b1ad',
'stage5_q_abb81ea1':'stage5_q_8fe38324',
'stage5_q_68b0a606':'stage5_q_a68df2d8',
'stage5_q_883d86a1':'stage5_q_5d1214bc',
'stage5_q_d4a2cbe5':'stage5_q_36d7acf4',
'stage5_q_362c3b34':'stage5_q_4054763b',
'stage5_q_0eef678d':'stage5_q_b22fc028',
'stage5_q_20ea0ede':'stage5_q_080da9a9',
'stage5_u_37d7c86a':'stage5_u_26770405',
'stage5_v_01bdd7f8':'stage5_v_b293b9ef',
'stage5_v_f8d9b122':'stage5_v_1f1a2261',
'battle_reaper_c84da985':'battle_reaper_85340834',

'stage6_b_26015c0f':'stage6_b_6f7a0ce3',
'stage6_d_822e73a6':'stage6_d_39950d4a',
'stage6_d_963d539c':'stage6_d_1837910c',
'stage6_e_59310e2f':'stage6_e_bce17fa9',
'stage6_e_d3f259c9':'stage6_e_dc031226',
'stage6_g_a2abaff3':'stage6_g_d66b4492',
'stage6_h_065865eb':'stage6_h_d893eea4',
'stage6_k_1f5b5300':'stage6_k_711295fe',
'stage6_k_1eef0538':'stage6_k_4a9d257e',
'stage6_k_16c2e038':'stage6_k_ae5d198f',
'stage6_k_c44970c1':'stage6_k_8bbf33f2',
'stage6_k_aefb39d8':'stage6_k_e26829f4',
'stage6_k_9f054503':'stage6_k_df7ec25b',
'battle_valkyrie_c0dc48f2':'battle_valkyrie_c0512d8c',
'battle_valkyrie_40c4062b':'battle_valkyrie_ef09200b',
'battle_valkyrie_4680d252':'battle_valkyrie_08071df7',
'battle_valkyrie_35a06076':'battle_valkyrie_265045a3',
'battle_valkyrie2_32847a27':'battle_valkyrie2_68bc6c77',

'stage7_0_6c74c2c7':'stage7_0_1a62630e',
'stage7_0_0b0d04b3':'stage7_0_3a7445c2',
'stage7_0_9be2b7d2':'stage7_0_0474ef7c',
'stage7_0_d97eb18e':'stage7_0_af14e992',
'stage7_0_a4b01e17':'stage7_0_4fd4bc86',
'stage7_0_37095bea':'stage7_0_8e935c7c',
'stage7_0_2b00e67a':'stage7_0_56217919',
'stage7_0_b182849a':'stage7_0_4b7a03d6',
'stage7_0_5a2b5e02':'stage7_0_b063cb4e',
'stage7_0_321dcc21':'stage7_0_a659cd62',
'stage7_0_06f3d25f':'stage7_0_1c0d9c2b',
'stage7_0_5d204f40':'stage7_0_8375fe7a',
'stage7_0_35c2f377':'stage7_0_642da8e6',
'stage7_0_0576fc5a':'stage7_0_dd7839b8',
'stage7_0_992919ec':'stage7_0_e341e9c6',
'stage7_0_0fda3f4f':'stage7_0_4908e71d',
'stage7_0_46798f34':'stage7_0_186f9dc5',
'stage7_0_eef0aa0c':'stage7_0_12a857ee',
'stage7_0_31ed6ad5':'stage7_0_d9a6e98e',
'stage7_0_bfd0b6b5':'stage7_0_9f4f3572',
'stage7_0_f20d1945':'stage7_0_fd971c27',
'stage7_0_9e315271':'stage7_0_b1cfaf47',
'stage7_0_d7c21988':'stage7_0_d3d434d3',
'stage7_0_152aeb88':'stage7_0_56748e86',
'stage7_a_6f55fe63':'stage7_a_af14d69e',
'stage7_a_c0fa84e7':'stage7_a_4b874647',
'stage7_a_721b4f4a':'stage7_a_6fbb4f8a',
'stage7_a_404f82ee':'stage7_a_74b06202',
'stage7_a_6e2f80ec':'stage7_a_434a457c',
'stage7_a_e86a86e1':'stage7_a_551c9c75',
'stage7_a_cdf83fce':'stage7_a_e3ea5039',
'stage7_g_b8772766':'stage7_g_56af7568',
'stage7_g_5cc877ac':'stage7_g_e19dc788',
'stage7_h_eda533fa':'stage7_h_78bd2d79',
'stage7_k_85c3c342':'stage7_k_29156850',
'stage7_k_128cf430':'stage7_k_b255109a',
'stage7_n_aeb71594':'stage7_n_d29f3797',
'stage7_n_128cf430':'stage7_n_b26ec5a4',
'stage7_n_f04a2738':'stage7_n_fcda7a46',
'stage7_n_821de08a':'stage7_n_c5390f80',
'stage7_p_f70b5d85':'stage7_p_26770405',
'stage7_q_923ae379':'stage7_q_152a960f',
'stage7_q_d1a5323b':'stage7_q_36a81a92',
'stage7_q_0ccb083a':'stage7_q_b4fa65bb',
'stage7_q_ce9d4c57':'stage7_q_616dccaa',
'stage7_q_e954806a':'stage7_q_231d16ce',
'stage7_q_56cccf0f':'stage7_q_e3060ada',
'stage7_q_2b524b6f':'stage7_q_041c2a59',
'stage7_q_e632988a':'stage7_q_40e91408',
'stage7_q_e30e2452':'stage7_q_d7700b82',
'stage7_q_1cac5925':'stage7_q_d02931bd',
'stage7_q_759d0b09':'stage7_q_23961c14',
'stage7_q_96f8a949':'stage7_q_3f9213d4',
'stage7_q_fc01ee37':'stage7_q_8015b73d',
'stage7_q_82c29c3e':'stage7_q_98b9b110',
'stage7_q_04013316':'stage7_q_fe4793c8',
'stage7_q_5569ec1d':'stage7_q_c7726884',
'stage7_q_a7fd5eee':'stage7_q_ac187839',
'stage7_q_81b15f49':'stage7_q_f0650bfa',
'stage7_q_8312c992':'stage7_q_74952e13',
'stage7_u_a43db312':'stage7_u_9002f9a8',
'stage7_u_0f99ff2b':'stage7_u_0034f72c',
'stage7_u_5e20ebf5':'stage7_u_ca3acaf4',
'stage7_u_12def71d':'stage7_u_66f8ccc9',
'stage7_u_c322de03':'stage7_u_700839f8',
'stage7_u_2beee33f':'stage7_u_6ff1ec2f',
'stage7_u_36c5c511':'stage7_u_af625aea',
'stage7_u_e2f84ad5':'stage7_u_9e6f9816',
'stage7_u_412f7248':'stage7_u_62ee1ec4',
'stage7_v_05f37a55':'stage7_v_6a750f2e',

'stage8_a_ccdb36f1':'stage8_a_1579de59',
'stage8_b_252d271f':'stage8_b_3116ce09',
'stage8_b_2d638a72':'stage8_b_89ad0542',
'stage8_b_c44ecdbe':'stage8_b_f04a8187',
'stage8_b_1595b325':'stage8_b_83b20f41',
'stage8_b_4c2dcd78':'stage8_b_72655543',
'stage8_b_f93d873c':'stage8_b_0b6798ce',
'stage8_b_4ff298e9':'stage8_b_c12592c8',
'stage8_b_0b017096':'stage8_b_351f048e',
'stage8_c_1adcae97':'stage8_c_28437008',
'stage8_c_9741e4bb':'stage8_c_27144df9',
'stage8_c_04c286c2':'stage8_c_e4e4946d',
'stage8_c_07e5b3f8':'stage8_c_c969691a',
'stage8_c_705a3470':'stage8_c_fd8ba986',
'stage8_c_cadf0d14':'stage8_c_de629eb9',
'stage8_c_99864952':'stage8_c_ae4f5509',
'stage8_c_8090ebee':'stage8_c_68e252b7',
'stage8_boss_74223f91':'stage8_boss_4e70c20f',
'stage8_boss_6f63617f':'stage8_boss_b38c4d7a',
'stage8_boss_fd3f70d4':'stage8_boss_288edf1e',
'stage8_boss_06203e41':'stage8_boss_bd7bc99d',
'stage8_boss_20736d6e':'stage8_boss_3dc33efe',
'stage8_boss_b28de181':'stage8_boss_e64fc015',
'stage8_boss_1c8165dd':'stage8_boss_4ad85908',
'stage8_boss_ae85c71b':'stage8_boss_8011efe8',
'stage8_boss2_d1a8a418':'stage8_boss2_c4f2b6da',
'stage8_boss2_dcd263e1':'stage8_boss2_a46a0612',
'stage8_boss2_9ac0456b':'stage8_boss2_2a2cdb51',
'stage8_boss2_21f4263e':'stage8_boss2_e746a945',
'stage8_boss2_1a34cb6f':'stage8_boss2_4eac3703',
'stage8_boss2_2a63da6f':'stage8_boss2_cfec1113',
'stage8_boss2_294b6cd0':'stage8_boss2_b69ae26d',
'stage8_boss2_5a8aece0':'stage8_boss2_83d01907',
'stage8_boss2_b0ae2c86':'stage8_boss2_daf7a272',
'stage8_boss2_7e673f74':'stage8_boss2_69bf3a19',
'stage8_boss2_c5c6ce3e':'stage8_boss2_702a685a',
'stage8_boss2_26d360cd':'stage8_boss2_beba145a',
'stage8_boss2_845612b4':'stage8_boss2_a3d7ed72',
'stage8_boss2_ee169d1f':'stage8_boss2_a5e37f7f',
'stage8_boss2_aafabec2':'stage8_boss2_304ce920',
'stage8_boss2_f11476fb':'stage8_boss2_ff07ff32',
'stage8_boss2_12ffa28e':'stage8_boss2_56531135',
'stage8_boss2_cf235af9':'stage8_boss2_a5da4aac',
'stage8_boss2_5cfd806d':'stage8_boss2_c780acc7',
'stage8_boss2_7a9ac16b':'stage8_boss2_63a93730',
'stage8_boss2_924a0cc8':'stage8_boss2_6875cebd',
'stage8_boss2_e3223922':'stage8_boss2_d498b828',
'stage8_boss2_86a8f2e7':'stage8_boss2_e5e382e9',
'stage8_boss2_e2eb45d6':'stage8_boss2_60d623d6',
'stage8_boss2_e8c8c332':'stage8_boss2_1cd320de',
'stage8_boss2_6dbba420':'stage8_boss2_408d87bf',
'stage8_boss2_012b9a88':'stage8_boss2_0a5a1574',
'stage8_boss2_2343d315':'stage8_boss2_067556d5',
'stage8_boss2_de581f71':'stage8_boss2_81af2bcf',
'stage8_boss2_7b8d8e90':'stage8_boss2_948510cc',
'stage8_boss2_6471473c':'stage8_boss2_f4ae32e9',
'stage8_boss2_fe266ec5':'stage8_boss2_49b3b8d3',
'stage8_boss2_3d753509':'stage8_boss2_13195310',
'stage8_boss2_32997c86':'stage8_boss2_953d5011',
'stage8_boss2_5020f6e8':'stage8_boss2_8dba9559',
'stage8_boss2_a438e56b':'stage8_boss2_1ce30b3f',
'stage8_boss2_81b24e7d':'stage8_boss2_dea1e978',
'stage8_boss2_fe0a42f9':'stage8_boss2_c763738b',
'stage8_boss2_f84f29af':'stage8_boss2_458e80d8',
'stage8_boss2_9bb3a5f7':'stage8_boss2_cd6afea3',
'stage8_boss2_f64d3ad5':'stage8_boss2_7baf635a',
'stage8_boss2_b372a937':'stage8_boss2_42f4b654',
'stage8_boss2_d61e249f':'stage8_boss2_9654d561',
'stage8_boss2_aa19d9e6':'stage8_boss2_c47e5f9b',
'stage8_boss2_0724894d':'stage8_boss2_7620202d',
'stage8_boss2_e02f983a':'stage8_boss2_6fe1c684',
'stage8_boss2_99258800_1':'stage8_boss2_99258800',
'stage8_boss2_d1a8a418_1':'stage8_boss2_c4f2b6da',
'stage8_boss2_dcd263e1_1':'stage8_boss2_a46a0612',
'stage8_boss2_9ac0456b_1':'stage8_boss2_2a2cdb51',
'stage8_boss2_203b7a3f':'stage8_boss2_b883da65',
'stage8_boss2_7774e3dc':'stage8_boss2_1140fae4',
'stage8_boss2_9a987c17':'stage8_boss2_ec7ac9a1',
'stage8_boss2_0fbacca5':'stage8_boss2_725c0902',
'stage8_boss2_4c652f71':'stage8_boss2_9f907dd1',
'stage8_boss2_67c22bda':'stage8_boss2_12e95b7c',
'stage8_boss2_1d174f3d':'stage8_boss2_292cacc3',
'stage8_boss2_2bcdd87d':'stage8_boss2_e198cce1',
'stage8_boss2_de73b24b':'stage8_boss2_fc71b5ff',
'stage8_boss2_b2ccb132':'stage8_boss2_33304948',
'stage8_boss2_36a9241b':'stage8_boss2_3420b614',
'stage8_boss2_dc4f13f4':'stage8_boss2_8d951fc2',
'stage8_boss2_20095291':'stage8_boss2_daf7a272',
'stage8_boss2_9db6caa2':'stage8_boss2_e6882287',
'stage8_boss2_65ad17aa':'stage8_boss2_3420b614',
'stage8_boss2_1849e59a':'stage8_boss2_5b0b10a4',
'stage8_boss2_25c66105':'stage8_boss2_41953890',
'stage8_boss2_753a8f61':'stage8_boss2_97dbe7f4',
'stage8_boss2_b29d3666':'stage8_boss2_4c2302b9',
'stage8_boss2_539a7e69':'stage8_boss2_7fb3d4c5',
'stage8_boss2_70c2a463':'stage8_boss2_9440aad7',
'stage8_boss2_642f544f':'stage8_boss2_004c1b78',
'stage8_boss2_996da2c0':'stage8_boss2_d22c4d0b',
'stage8_boss2_6e60ca8a':'stage8_boss2_e464a2ba',
'stage8_boss2_35b73adc':'stage8_boss2_b3e9735b',
'stage8_boss2_23f16222':'stage8_boss2_f3806847',
'stage8_boss2_82396ba2':'stage8_boss2_18372668',
'stage8_boss2_0b09d4a7':'stage8_boss2_7965c382',
'stage8_boss2_403cfd4b':'stage8_boss2_e3d73ea5',
'stage8_boss2_ad1f9d14':'stage8_boss2_f8af4529',
'stage8_boss2_312db793':'stage8_boss2_9a366f6d',
'stage8_boss2_d0f08b63':'stage8_boss2_60c39eb8',
'stage8_boss2_691399de':'stage8_boss2_79a7b55c',
'stage8_boss2_50dbb07e':'stage8_boss2_f7918a15',
'stage8_boss2_e903513f':'stage8_boss2_ca96ed85',
'stage8_boss2_4e205f93':'stage8_boss2_921068f7',
'stage8_boss2_aabb99eb':'stage8_boss2_69568631',
'stage8_boss2_bc944898':'stage8_boss2_dc4f2a24',
'stage8_boss2_2eb92a9d':'stage8_boss2_6d430d19',
'battle_levia2_24ce3a19':'battle_levia2_b5ec5459',
'battle_levia2_cdd6e025':'battle_levia2_85f63fff',
'battle_levia3_8a50f4d0':'battle_levia3_7bc9cd1e',
'battle_levia3_80d689ff':'battle_levia3_5ecc071d',
'battle_levia4_2406bd96':'battle_levia4_c0fcccc4',
'battle_levia4_57c8cf83':'battle_levia4_b654c3f6',
'battle_levia4_accbe4bd':'battle_levia4_c02afbea',
'battle_levia4_8550fb7e':'battle_levia4_f98293d6',
'battle_levia4_46c6b398':'battle_levia4_bacc40e0',
'battle_levia4_f962854b':'battle_levia4_85f63fff',

'stage9_b_88e14b69':'stage9_b_d4d4275a',
'stage9_c_1bde46ba':'stage9_c_923c9daa',
'stage9_d_1a13cfb1':'stage9_d_9024274d',
'stage9_e_cd3a6458':'stage9_e_e109e519',
'stage9_g_ccdb36f1':'stage9_g_1579de59',

'stage10_a_30b3a744':'stage10_a_16b71133',
'stage10_b_c72a3b96':'stage10_b_78bd2d79',
'stage10_c_37d7c86a':'stage10_c_26770405',
'stage10_d_d74aaa44':'stage10_d_f33e84b6',
'stage10_e_8a54161e':'stage10_e_14b7d1a9',
'stage10_g_065865eb':'stage10_g_d893eea4',
'stage10_h_6dde26b5':'stage10_h_4ac4e84f',
'stage10_h_549123e0':'stage10_h_e13add70',
'stage10_h_40931b68':'stage10_h_e20a2dfe',
'stage10_h_231b76f1':'stage10_h_269f66a0',
'stage10_h_3134ab08':'stage10_h_8499d3e1',

'event6_d0f10958':'event6_d129a22b',
'event6_12538c3a':'event6_383123fc',
'event6_f08b2189':'event6_19f1c67b',
'event6_9e502161':'event6_68e5e0d8',
'event6_35cd097d':'event6_a2330be3',
'event6_01c49dfd':'event6_0b1e39b7',
'event6_0081d626':'event6_99d91e33',
'event6_66360a22':'event6_d5826437',
'event6_d7bffd6c':'event6_d7bffd6c',
'event6_49e48e5d':'event6_0aff080b',
'event6_0ecd0630':'event6_09e65010',
'event6_25c92314':'event6_f2ee8b8f',
'event10_fd944171':'event10_ce46448d',

'ending1_2cbe785f':'ending1_98a56bb6',
'ending1_1b1c0773':'ending1_1f31624e',
'ending1_eb0434ac':'ending1_c98e77ab',
'ending1_28a51591':'ending1_5681b705',
'ending1_6ad41b49':'ending1_248585b8',
'ending1_08236fca':'ending1_8c5995e0',
'ending1_fe21a743':'ending1_ce815253',
'ending1_9dca774d':'ending1_74b20395',
'ending1_4b831f51':'ending1_b4702feb',
'ending1_d6893401':'ending1_7e8539cc',
'ending1_be0b231c':'ending1_fdf7916b',

'ending2_0551de38':'ending2_47f25d82',
'ending2_32742b17':'ending2_2a56f372',
'ending2_2fc30c67':'ending2_dae904a9',
'ending2_5d2f268b':'ending2_bd389f7f',
'ending2_42d86ff9':'ending2_d1666100',
'ending2_34f3e788':'ending2_e57fe474',
'ending2_4b66181e':'ending2_f3d71cf7',
'ending2_8bd4035a':'ending2_b08b5a11',
'ending2_1a3f7138':'ending2_fb3b3ae6',
'ending2_c2437977':'ending2_91d8f021',
'ending2_fc3640d2':'ending2_2dbd7318',
'ending2_55036316':'ending2_1a62630e'
}